;;;-*- Mode: Lisp; Package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-styling.lisp
;;;      
;;;      copyright  2008 Glen Foy, all rights reserved,
;;;
;;;      Styling functions for the incremental and batch algorithms.
;;;
;;; ****************************************************************************

(in-package "CC")

(defParameter *static-hash-table* (make-hash-table :test 'equal))

;;; Generate styling functions based on the defstyle descriptions.
(defun construct-function (form-name function-list add-to-hash-table-p &optional (table *static-hash-table*))
   (let* ((symbol-style-accessor-name (concatenate 'string form-name "-SYMBOL-STYLE"))
          (symbol-style-accessor (intern (string-upcase 
                                          (concatenate 'string form-name "-SYMBOL-STYLE")) :cc))
          (macro-style-accessor (intern (string-upcase 
                                         (concatenate 'string form-name "-STYLE")) :cc))
          (form-function (concatenate 'string form-name "-STYLING-FUNCTION"))
          (code (format nil "(defun ~a (form-start) (let ((form-end (sexp-end form-start)) (position form-start) 
(symbol-style-accessor-name ~s) (symbol-style-accessor '~s) (macro-style-accessor '~s))
(declare (optimize (speed 3) (safety 0)) (type (fixnum position)))" 
                         form-function symbol-style-accessor-name symbol-style-accessor macro-style-accessor)))
     (dolist (function function-list)
        (setf code (format nil "~a ~s" code (funcall function))))
     (setf code (format nil "~a))" code))
     (let ((ccl::*suppress-compiler-warnings* t)) ; muffle unused var messages
       (if add-to-hash-table-p
         (setf (gethash (string-upcase form-name) table) (eval (read-from-string code)))
         (eval (read-from-string code))))))

(defun get-function (table name)
   (gethash (string-upcase name) table))

;;; ----------------------------------------------------------------------------
;;; Functions called from construct-function
;;; ----------------------------------------------------------------------------
;;;
(defun superparen ()
    "Super parens surround top-level forms and embedded function definitions."
  `(prog ()
     (when position
       (unless (> (1+ position) (buffer-size *buffer*))
         (if *dynamic-p*
           (let* ((code-list (fcodes (superparen-style)))
                  (ff (first code-list))
                  (ms (second code-list)))
             (buffer-set-font-codes *buffer* ff ms 
                                    position (1+ position)))
           (style-region (superparen-style) position (1+ position)))
         (setf position (next-sexp-start (1+ position)))))))

(defun eval-when-superparen ()
   "Eval-when deserves a distinctive style for its parens."
   `(prog ()
       (when position
          (unless (> (1+ position) (buffer-size *buffer*))
             (if *dynamic-p*
               (let* ((code-list (fcodes (eval-when-superparen-style)))
                      (ff (first code-list))
                      (ms (second code-list)))
                 (buffer-set-font-codes *buffer* ff ms 
                                        position (1+ position)))
               (style-region (eval-when-superparen-style) position (1+ position)))
             (setf position (next-sexp-start (1+ position)))))))

(defun loop-superparen ()
   "Loop deserves a distinctive style for its parens."
   `(prog ()
       (when position
          (unless (> (1+ position) (buffer-size *buffer*))
             (if *dynamic-p*
               (let* ((code-list (fcodes (loop-superparen-style)))
                      (ff (first code-list))
                      (ms (second code-list)))
                 (buffer-set-font-codes *buffer* ff ms 
                                        position (1+ position)))
               (style-region (loop-superparen-style) position (1+ position)))
             (setf position (next-sexp-start (1+ position)))))))

(defun paren ()
    "This does no styling; it just increments POSITION."
  `(prog ()
     (when position (setf position (next-sexp-start (1+ position))))))

(defun optional-paren ()
    "This does no styling; it just increments POSITION, if there is a paren."
  `(prog ()
     (when (and position
                (or (char= (buffer-char *buffer* position) #\()
                    (char= (buffer-char *buffer* position) #\))))
       (setf position (next-sexp-start (1+ position))))))

(defun symbol ()
    "Style a symbol-name, taking into account exported symbols."
  `(prog ()
     (when position
       (cond (*do-exported-symbols*
              (let ((name (string-upcase 
                           (buffer-substring *buffer* position (sexp-end position)))))
                (when name
                  (multiple-value-bind (symbol kind)
                                       (find-symbol name *current-package*)
                    (cond ((and symbol *current-package* (eq kind :external)
                                (not (eq (funcall symbol-style-accessor)
                                         (variable-definition-symbol-style))))
                           (cond ((char= (buffer-char *buffer* position) #\") ; a string, don't set caps  
                                  (style-region (exported-symbol-style) 
                                                position (sexp-end position) nil))
                                 (t
                                  (style-region (exported-symbol-style)
                                                position (sexp-end position)))))                                  
                          (t ; not exported or a variable def
                           (cond ((char= (buffer-char *buffer* position) #\")
                                  (style-region (funcall symbol-style-accessor)
                                                position (sexp-end position) nil))
                                 (t
                                  (style-region (funcall symbol-style-accessor)
                                                position (sexp-end position))))))))))
             (t
              (cond ((char= (buffer-char *buffer* position) #\")
                     (style-region (funcall symbol-style-accessor) position (sexp-end position) nil))
                    (t
                     (cond (symbol-style-accessor
                            (style-region (funcall symbol-style-accessor) position (sexp-end position)))
                           (t
                            (ed-beep)
                            (format t "~%missing: ~s" symbol-style-accessor-name)
                            (break)
                            ))))))
       
       (setf position (next-sexp-start (1+ (sexp-end position)))))))

(defun struct-sym ()
    "Style the name of a structure."
  `(prog ()
     (when position
       (let ((end (sexp-end position)))
         (when end
            (struct-sym-styling-function position)
            ;; *** redundant
            (setf position (next-sexp-start (1+ end))))))))

(defun struct-fields ()
    "Style structure fields."
  `(prog ()
     (when position
       (do* ((field-start position (next-sexp-start (1+ field-end)))
             (field-end (when field-start (sexp-end field-start))
                        (when field-start (sexp-end field-start))))
            ((or (null field-start) (> field-start form-end)))
         (cond ((char= (buffer-char *buffer* field-start) #\()
                (let ((symbol-start (1+ field-start)))
                  (style-region (defstruct-field-style) symbol-start (sexp-end symbol-start))
                  (when (next-sexp-start (sexp-end symbol-start))
                    (style-buffer (next-sexp-start (sexp-end symbol-start)) field-end))))
               (t
                (style-region (defstruct-field-style) field-start (sexp-end field-start)))))
       (setf position (1- form-end)))))

(defun ancestor ()
    "Style a structure's ancestor."
  `(prog ()  
     (when position
       (let* ((start (next-sexp-start (1+ position)))
              (end (when start (sexp-end start)))
              (string (when (and start end) (buffer-substring *buffer* start end))))
         (when (and string (string-equal string ":include"))
           (style-region (keyword-package-style) start end)
           (when (next-sexp-start (1+ end))
             (style-region (defstruct-ancestor-style) (next-sexp-start (1+ end))
                           (sexp-end (next-sexp-start (1+ end)))))
           (setf position (next-sexp-start (1+ (sexp-end position)))))))))

(defun macro ()
    "Style the name of the macro."
  `(prog ()
     (when position
       (style-region (funcall macro-style-accessor) position (sexp-end position))
       (setf position (next-sexp-start (1+ (sexp-end position)))))))

(defun derivation-list ()
  "Style the DEFCLASS derivation list."
  `(prog ()
     (when position
       (style-region (defclass-derivation-style) (1+ position) (1- (sexp-end position)))
       (setf position (next-sexp-start (1+ (sexp-end position)))))))

(defun slot-list ()
  "Style DEFCLASS slots."
  `(prog ()
     (when position
       (let (slot-positions
             (end (sexp-end position)))
         (do* ((current-start (buffer-bwd-sexp *buffer* (1- end))
                              (buffer-bwd-sexp *buffer* (1- current-start))))
              ((<= current-start position))
           (when (or (not *dynamic-p*)
                     (and *dynamic-p*
                          (>= *dynamic-pos* current-start)
                          (<= *dynamic-pos* (buffer-fwd-sexp *buffer* current-start))))
             (push current-start slot-positions)))
         (dolist (slot-position slot-positions)
           (style-buffer slot-position (sexp-end slot-position))
           (style-region (defclass-slot-style) (1+ slot-position)
                         (sexp-end (1+ slot-position))))
         (setf position (next-sexp-start (1+ end)))))))

(defun qualifier ()
    "Style method qualifiers."
  `(prog ()
     (when position
       (when (char= (buffer-char *buffer* position) #\:)
         (style-region (keyword-package-style) position (sexp-end position))
         (setf position (next-sexp-start (1+ (sexp-end position))))))))

(defun list-regions (start end  &aux e1-start e1-end e2-start e2-end)
    "List parameter and specializer or optional parameter and defaults."
  (setf e1-end (buffer-fwd-sexp *buffer* (1+ start))
        e1-start (buffer-bwd-sexp *buffer* e1-end))
  (setf e2-start (buffer-bwd-sexp *buffer* (1- end))
        e2-end (buffer-fwd-sexp *buffer* e2-start))
  (list e1-start e1-end e2-start e2-end))

(defun parameter-regions (list-start)
   "Collect specialized and non-specialized parameter regions. Style the defaults for
lambda-list-keyword parameters."
   (let ((list-end (sexp-end list-start))
         results option-p)
     (do* ((start (next-sexp-start (1+ list-start)) 
                   (when (sexp-end start) (next-sexp-start (1+ (sexp-end start)))))
            (char (buffer-char *buffer* start) (when start (buffer-char *buffer* start))))
           ((or (null start) (>= start list-end)) results)
        (cond ((char= char #\()
                (let ((specializer-regions (list-regions start (sexp-end start))))
                  (when (and option-p (third specializer-regions) (fourth specializer-regions))
                     (style-buffer (third specializer-regions) (fourth specializer-regions)))
                  (push (subseq specializer-regions 0 (when option-p 2))
                         results)))
               ((char= char #\&) 
                (style-region (keyword-package-style) start (sexp-end start))
                (setf option-p t))
               (t 
                (push (list start (sexp-end start)) results))))))

(defun parameter-list ()
    "Style the parameter list.  This is called by both functions and methods."
  `(prog ()
     (when position
       (let ((parameter-regions (parameter-regions position)))
         (dolist (arg parameter-regions)
           (style-region (parameter-style) (first arg) (second arg))
           (when (and (third arg) (fourth arg))
             (style-region (specializer-style) (third arg) (fourth arg))))
         (setf position (next-sexp-start (1+ (sexp-end position))))))))

(defun embedded-function-definitions ()
    "Style the functions defined by LABELS and FLET."
  `(prog ()
     (when position
       (let ((end (sexp-end position)))
         (do ((position (next-sexp-start (1+ position)) (next-sexp-start (1+ (sexp-end position)))))
             ((or (null position) (>= position end)))
           (embedded-function-styling-function position))
         (setf position (next-sexp-start (1+ end)))))))

(defun variable-definitions ()
    "Style the variables and default values defined by LET, DO*, etc."
  `(prog ()
     (when position
       (let ((end (sexp-end position)))
         (do ((position (when position (next-sexp-start (1+ position))) 
                        (when position (next-sexp-start (1+ (sexp-end position))))))
             ((or (null position) (>= position end)))
           (variable-definition-styling-function position))
         (setf position (next-sexp-start (1+ end)))))))

(defun case-match-forms ()
    "Style the match forms of a case statement"
  `(prog ()
     (when position
       (let ((end (1- form-end)))
         (do ((position (when position (next-sexp-start position))
                        (when position (next-sexp-start (1+ (sexp-end position))))))
             ((or (null position) (>= position end)))
           (case-match-styling-function position))
         (setf position (next-sexp-start (1+ end)))))))

(defun loop-test ()
   "Style the test form used by an iteration macro."
   `(prog ()
       (when position
          (let ((end (sexp-end position)))
            (style-buffer position end)
            (setf position (next-sexp-start (1+ end)))))))

(defun variable-form ()
    "Style the initialization form of a variable definition."
  `(prog ()
     (when position
       (let ((end (sexp-end position)))
         (variable-definition-styling-function position)
         (setf position (next-sexp-start (1+ end)))))))

(defun variable-list ()
   "Style the variable list of multiple-value-setq, multiple-value-bind, etc."
   `(prog ()
       (when position
          (do* ((list-end (sexp-end position))
                 (var-start (next-sexp-start (1+ position)) (next-sexp-start (1+ var-end)))
                 (var-end (when var-start (sexp-end var-start))
                          (when var-start (sexp-end var-start))))
                ((or (null var-start) (> var-start list-end)))
             (style-region (variable-definition-symbol-style) var-start var-end nil))
          (setf position (next-sexp-start (1+ (sexp-end position)))))))

(defun body ()
  "Style the body of a macro."
  `(prog ()
     (when position
       (style-buffer position (1- form-end))
       (setf position (1- form-end)))))

(defun loop-body ()
  "Style the body of a loop macro."
  `(prog ()
     (when position
       (style-loop-elements position (1- form-end))
       (setf position (1- form-end)))))

(defun form ()
  "Style a single form."
  `(prog ()
     (when position
       (style-buffer position (sexp-end position))
       (setf position (if (next-sexp-start (1+ (sexp-end position)))
                        (min (1- form-end)
                             (next-sexp-start (1+ (sexp-end position))))
                        (1- form-end))))))

(defun doc ()
   "Style the doc in DEFUN, DEFMETHOD, DEFMACRO, DEFPARAMETER, etc."
   `(prog ()
      (when position
        (cond ((< position form-end)
               (cond ((char-equal #\" (buffer-char *buffer* position))
                      (cond (*dynamic-p*
                             (style-region (doc-style) 
                                           position (min *dynamic-pos* 
                                                         (sexp-end position)) nil))
                            (t
                             (style-region (doc-style) 
                                           position (sexp-end position) nil)))
                      (setf position (if (next-sexp-start (1+ (sexp-end position)))
                                       (min (1- form-end)
                                            (next-sexp-start (1+ (sexp-end position))))
                                       (1- form-end))))
                     (t
                      position)))
              (t 
               form-end)))))

(defun style-elements (symbol-start form-end)
   "Step through the code sexp by sexp, styling appropriately."
   (flet ((not-char-constant-p (element-start)
            (or (< element-start 2)
                 (char/= (buffer-char *buffer* (- element-start 1)) #\\)
                 (char/= (buffer-char *buffer* (- element-start 2)) #\#)))
          (check-dynamic-p (element-start element-end)
             (or (not *dynamic-p*)
                  (and *dynamic-p*
                        (>= *dynamic-pos* element-start)
                        (<= *dynamic-pos* element-end)))))    
     (do* ((element-start symbol-start
                           (when element-end (next-sexp-start element-end)))
            (element-end (when element-start (sexp-end element-start))
                         (when element-start (sexp-end element-start)))
            (current-char (when element-start (buffer-char *buffer* element-start))
                          (when element-start (buffer-char *buffer* element-start))))
           ((or (null element-start) (null element-end) (> element-start form-end)))
        (when (or (not *segment-array*)
                    (not-embedded-in-segment-p *segment-array* element-start))
           (when (or (char= current-char #\')
                       (char= current-char #\`)
                       (char= current-char #\,))
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\@)
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\')
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\,)
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (cond ((and (char= current-char #\()
                         (not-char-constant-p element-start)
                         (check-dynamic-p element-start element-end))
                   ;; this can make an indirect recursive call
                   (style-buffer element-start element-end)) ; lists
                  ((and (char= current-char #\#)
                         (< element-start (- (buffer-size *buffer*) 2))
                         (char= (buffer-char *buffer* (+ element-start 1)) #\')
                         (char= (buffer-char *buffer* (+ element-start 2)) #\()
                         (check-dynamic-p element-start element-end))
                   (style-buffer element-start element-end)) ; lambdas
                  ((and *do-keyword-package*
                         (char= current-char #\:)
                         (not-char-constant-p element-start))
                   (style-region (keyword-package-style) ; keywords
                                 element-start (sexp-end element-start))))))))

(defun style-loop-elements (symbol-start form-end)
   "Step through the code sexp by sexp, styling appropriately."
   (flet ((not-char-constant-p (element-start)
            (or (< element-start 2)
                 (char/= (buffer-char *buffer* (- element-start 1)) #\\)
                 (char/= (buffer-char *buffer* (- element-start 2)) #\#)))
          (check-dynamic-p (element-start element-end)
             (or (not *dynamic-p*)
                  (and *dynamic-p*
                        (>= *dynamic-pos* element-start)
                        (<= *dynamic-pos* element-end)))))    
     (do* ((element-start symbol-start
                           (when element-end (next-sexp-start element-end)))
            (element-end (when element-start (sexp-end element-start))
                         (when element-start (sexp-end element-start)))
            (current-char (when element-start (buffer-char *buffer* element-start))
                          (when element-start (buffer-char *buffer* element-start))))
           ((or (null element-start) (null element-end) (> element-start form-end)))
        (when (or (not *segment-array*)
                    (not-embedded-in-segment-p *segment-array* element-start))
           (when (or (char= current-char #\')
                       (char= current-char #\`)
                       (char= current-char #\,))
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\@)
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\')
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (when (char= current-char #\,)
              (incf element-start)
              (setf current-char (buffer-char *buffer* element-start)))
           (cond ((and (char= current-char #\()
                         (not-char-constant-p element-start)
                         (check-dynamic-p element-start element-end))
                    (style-buffer element-start element-end)) ; lists
                   ((and (char= current-char #\#)
                          (< element-start (- (buffer-size *buffer*) 2))
                          (char= (buffer-char *buffer* (+ element-start 1)) #\')
                          (char= (buffer-char *buffer* (+ element-start 2)) #\()
                          (check-dynamic-p element-start element-end))
                    (style-buffer element-start element-end))
                   ((and *do-keyword-package*
                          (char= current-char #\:)
                          (not-char-constant-p element-start))
                    (style-region (keyword-package-style)
                                  element-start element-end))
                   ((and (alpha-char-p current-char) ; loop keyword?
                          (loop-keywd-p *rtn-grammar* (buffer-substring *buffer* element-start element-end)))
                    (style-region (loop-keywords-style) 
                                  element-start element-end)))))))

(defun options ()
    "Style DEFCLASS and DEFGENERIC options."
  `(prog ()
     (when position
       (do* ((option-start position (next-sexp-start (sexp-end option-start)))
             (symbol-start (when option-start (1+ option-start))
                           (when option-start (1+ option-start))))
            ((or (null symbol-start) (>= symbol-start form-end)))
         (when (char-equal #\: (buffer-char *buffer* symbol-start))
           (style-region (keyword-package-style) symbol-start (sexp-end symbol-start) nil)
           (cond ((string-equal (buffer-substring *buffer* symbol-start (sexp-end symbol-start))
                                ":documentation")
                  (when (next-sexp-start (sexp-end symbol-start))
                    (style-region (doc-style) 
                                  (next-sexp-start (sexp-end symbol-start))
                                  (sexp-end (next-sexp-start (sexp-end symbol-start))) nil)))
                 (t 
                  (when (next-sexp-start (sexp-end symbol-start))
                    (style-elements (next-sexp-start (sexp-end symbol-start)) form-end))))))
       (setf position (1- form-end)))))

(defun keyword-pairs ()
    "Step through the code sexp by sexp starting at POSITION, processing the entire form."
  `(prog ()
     (when position
       (style-elements position form-end)
       (setf position (1- form-end)))))

(defun forward-list (start &optional (end (buffer-size *buffer*)))
   "Get the next #\( that is not embedded in a comment and not a character constant."
   (when (or (null start) (null end)) (return-from forward-list nil))
   (do* ((next (ccl::buffer-forward-search *buffer* #\( start end)
                (ccl::buffer-forward-search *buffer* #\( (1+ next) end))
          not-embedded)
         ((null next) (return nil))
      ;; ***
      (if *segment-array*
        (setf not-embedded (not-embedded-in-segment-p *segment-array* (1- next)))
        (setf not-embedded t))
      (cond ((>= next 2)
              (when (and not-embedded
                            ;; buffer-forward-search return the position AFTER the target
                            (neq (buffer-char *buffer* (- next 2)) #\\)
                            (neq (buffer-char *buffer* (- next 3)) #\#)
                            (neq (buffer-char *buffer* (- next 2)) #\#)) ; |#(-reader| 
                 (return (1- next))))
             (t 
              (when not-embedded (return (1- next)))))))

(defun list-forms (start end)
    "Returns a list of starting positions for all the top-level lists in the range START, END."
  (do* ((positions nil)
        (sexp-start (cond ((char= (buffer-char *buffer* start) #\()
                           start)
                          (t
                           (next-sexp-start start)))
                    (forward-list sexp-end end))
        (sexp-end (when sexp-start (buffer-fwd-sexp *buffer* sexp-start end))
                  (when sexp-start (buffer-fwd-sexp *buffer* sexp-start end)))
        (current-char (when sexp-start (buffer-char *buffer* sexp-start))
                      (when sexp-start (buffer-char *buffer* sexp-start))))
       ((or (null sexp-end)
            (null sexp-start)
            (> sexp-start end))
        (return (nreverse positions)))
    (when (or (char= current-char #\')
              (char= current-char #\`)
              (char= current-char #\,))
      (incf sexp-start) 
      (setf current-char (buffer-char *buffer* sexp-start)))
    (when (char= current-char #\@)
      (incf sexp-start) 
      (setf current-char (buffer-char *buffer* sexp-start)))
    (when (char= current-char #\')
      (incf sexp-start) 
      (setf current-char (buffer-char *buffer* sexp-start)))
    (when (char= current-char #\,)
      (incf sexp-start) 
      (setf current-char (buffer-char *buffer* sexp-start)))
    ;; When styling dynamically, only want include forms 
    ;; if *dynamic-pos* is inside the form.
    (cond ((char= current-char #\()
           (when (or (not *dynamic-p*)
                     (and *dynamic-p*
                          (>= *dynamic-pos* sexp-start)
                          (<= *dynamic-pos* sexp-end)))
             (push sexp-start positions)))
          ((char= current-char #\#)
           (cond ((and (< sexp-start (1+ (buffer-size *buffer*)))
                       (char= (buffer-char *buffer* (+ sexp-start 1)) #\')
                       (char= (buffer-char *buffer* (+ sexp-start 2)) #\())
                  (when (or (not *dynamic-p*)
                            (and *dynamic-p*
                                 (>= *dynamic-pos* sexp-start)
                                 (<= *dynamic-pos* sexp-end)))
                    (push (+ sexp-start 2) positions))))))))

(defun form-styled-p (position)
   "If there is a stylable form style at POSITION, style it and return T.  If not, return NIL."
   (when position
      (let* ((symbol-start (1+ position))
             (symbol-end (sexp-end (1+ position)))
             (string (buffer-substring *buffer* symbol-start symbol-end))
             (styling-function (get-function *static-hash-table* string)))
        (when styling-function 
           (funcall styling-function position) 
           t))))

(defun package-form-styled-p (position)
   "If there is a :cl function at POSITION, style it and return T.  If not, return NIL."
   (when (and position
                 (or *do-cl-package* *do-keyword-package*))
      (let* ((symbol-start (1+ position))
             (symbol-end (sexp-end (1+ position))))
        (cond ((and *do-keyword-package* (char= (buffer-char *buffer* position) #\:))
                (style-region (keyword-package-style) symbol-start symbol-end) t)
               ((and *do-cl-package* 
                      (find-symbol 
                       (string-upcase (buffer-substring *buffer* symbol-start symbol-end)) :cl))
                (style-region (cl-package-style) symbol-start symbol-end)
                t)))))

(defun style-buffer (&optional (start 0) (end (buffer-size *buffer*)))
    "Style the buffer, given the range START, END."
  (let ((positions (list-forms start end))
        form-end)
    (cond (positions 
           (dolist (position positions)
             (unless (form-styled-p position)
               (when (or *do-cl-package* *do-keyword-package*)
                 (setf form-end (buffer-fwd-sexp *buffer* position end))
                 (cond ((package-form-styled-p position) 
                        (setf position (next-sexp-start (sexp-end (1+ position)))))
                       (t
                        (incf position)))
                 (when position (style-elements position form-end))))))
          (t
           (style-elements (1+ start) end)))))

;;; ----------------------------------------------------------------------------
;;; The static styling interface:
;;;
;;; (setf *rtn-batch-styling-p* t)
;;; (setf *rtn-batch-styling-p* nil)
;;;
(defMethod view-style-buffer ((window fred-window) &optional start end)
   (setf *current-package* (current-package))
   (setf *buffer* (fred-buffer window))
   (init-parse *parser*)
   (cond ((> (buffer-size *buffer*) 0)
            (unless (and start end)
               (multiple-value-setq (start end)
                  (selection-range window)))
            (set-batch-processing)
            (prog1 
               (if (= start end)
                 ;; This is not a pref. It is an internal flag used to do batch
                 ;; styling with the RTN algorithm.  The RTN is slower than the
                 ;; recursive descent algorithm in batch mode, so ordinarily it
                 ;; is not used.  Batch styling with the RTN is a way of 
                 ;; exercising the RTN neworks.
                 (if *rtn-batch-styling-p* 
                   (rtn-style-buffer :start 0 :end (buffer-size *buffer*))
                   (style-buffer))
                 (if *rtn-batch-styling-p*
                   (rtn-style-buffer :start start :end #'(lambda () end))
                   (style-buffer start end)))
               (fred-update window)))
           (t
            (ed-beep)
            (format t "~% Empty File ..."))))

;;; This is only called by the prefs dialog to update the sample code.
(defMethod view-style-buffer ((item fred-mixin) &optional start end)
  (setf *buffer* (fred-buffer item))
  (set-background-color item)
  (set-generic-text-style (fred-buffer item))
  (view-style-buffer-comments item start end)
  (style-buffer)
  (fred-update item))

(defMethod window-style-buffer ((fred fred-window))
   (handler-case
      (progn
         (setf *current-package* (current-package))
         (when (and *background-color*
                       (part-color fred :body)
                       (not (= (part-color fred :body)
                                 (background-color))))
            (set-part-color fred :body (background-color))
            (set-part-color fred :content (background-color)))
         
         (with-cursor *watch-cursor*
           (multiple-value-bind (start end)
                                  (selection-range fred)
              (cond ((= start end) ; no selection
                      (set-generic-text-style (fred-buffer fred))
                      (view-style-buffer-comments fred 0 (buffer-size (fred-buffer fred))))
                     (t
                      (set-generic-text-style (fred-buffer fred) start end)
                      (view-style-buffer-comments fred start end)))
              (view-style-buffer fred)
              (fred-update fred))))
      
      (ccl::modify-read-only-buffer 
       (c)
       (declare (ignore c))
       (when (ccl::buffer-whine-read-only (fred-item fred))
          (window-style-buffer fred)))))

(defun style-folder-recursively ()
  (let ((dir (choose-directory-dialog)))
    (cond ((pathnamep dir)
           (time
            (format t "~%~%~a files styled."
                    (style-folder (directory-namestring dir)))))
          (t
           (time
            (format t "~%~%~a files styled."
                    (style-folder dir)))))))

(defun style-folder (folder)
   (let ((files (directory (merge-pathnames folder "*.lisp") :files t :directories nil))
         (folders (directory (merge-pathnames folder "*") :files nil :directories t))
         (file-count 0)
         window)
     (dolist (file files)
        (format t "~%;;; styling: ~a" file)
        (incf file-count)
        ( setf window (ed file))
        (window-style-buffer window)
        (window-save window)
        (window-close window))
     (dolist (folder folders)
        (incf file-count (style-folder folder)))
     file-count))

(defMethod do-vanilla-style ((buffer buffer-mark) start end)
  ;; Set the font spec of the text to the default; but leave the capitalization
  ;; of strings and comments alone.
  (let (skip-list case)
    (with-cursor *watch-cursor*
      (buffer-set-font-spec buffer (f-spec '*vanilla-styling*)
                            start end)
      ;; *** this should use start and end
      (setf skip-list (get-combined-segment-list buffer))
      (setf case (c-spec '*vanilla-styling*))
      (cond (skip-list
             ;; set case, except in strings and comments or if :case equals :unchanged
             (do* ((size (buffer-size buffer))
                   (segment (pop skip-list) (pop skip-list))
                   (seg-start 0 next-start)
                   (seg-end (first segment) (first segment))
                   (next-start (second segment) (second segment)))
                  ((or (>= seg-start size)
                       (null seg-start)
                       (null seg-end)))
               (when (and (> (- seg-end seg-start) 1)
                          (>= seg-start start)
                          (<= seg-start end))
                 (cond ((eql case :up)
                        (buffer-upcase-region buffer seg-start 
                                              (min seg-end end)))
                       ((eql case :down)
                        (buffer-downcase-region buffer seg-start 
                                                (min seg-end end)))))))
            (t 
             (cond ((eql case :up)
                    (buffer-upcase-region buffer 0 (buffer-size buffer)))
                   ((eql case :down)
                    (buffer-downcase-region buffer 0 (buffer-size buffer)))))))))

(defMethod vanilla-style ((item scrolling-fred-view) start end)
  ;; Set the font spec of the text to the default; but leave the capitalization
  ;; of strings and comments alone.
  (handler-case
    (progn
      (set-part-color item :body *white-color*)
      (do-vanilla-style (fred-buffer item) start end)
      (fred-update item))
    (ccl::modify-read-only-buffer 
     (c)
     (declare (ignore c))
     (when (ccl::buffer-whine-read-only (fred-item item))
       (vanilla-style item start end)))))

(defMethod vanilla-style ((fred fred-window) start end)
   (handler-case
      (progn
         (set-part-color fred :body *white-color*)
         (let ((mb (view-mini-buffer fred)))
           (when mb 
              (set-part-color mb :body *white-color*)
              (invalidate-view mb)))
         (when *use-white-i-beam* (set-default-i-beam))
         (do-vanilla-style (fred-buffer fred) start end)
         (fred-update fred))
      (ccl::modify-read-only-buffer 
       (c)
       (declare (ignore c))
       (when (ccl::buffer-whine-read-only (fred-item fred))
          (vanilla-style fred start end)))))

;;; ed-indent-sexp calls this with values for start and end.
(defMethod ccl:ed-indent-for-lisp :before ((w window-fred-item) &optional start end)
  (let ((window (view-window w)))
    (unless (and start end)
      (multiple-value-setq (start end)
        (selection-range window)))
    (when (and *tab-key-styling* (not (= start end)) (typep window 'fred-window)) ; exclude pref dialog
      (with-cursor *watch-cursor*
        (set-generic-text-style (fred-buffer w) start end)
        (view-style-buffer-comments window start end)
        (view-style-buffer window start end)))))

;;; ----------------------------------------------------------------------------
;;; action functions and predicates called by the RTN networks
;;; ----------------------------------------------------------------------------
;;;
(defun start-action (pos env)
   "Estabilish the START and END positions for the form. End-type can be :form :parent or :subform"
   (when pos
      (unless (char= (buffer-char *buffer* pos) #\()
         (setf pos (next-sexp-start pos)))
      (let ((parent-end (get-parent-end env))
            end-type)
        ;; an unmatched quote will cause parent-end to be nil
        (when parent-end
           (cond ((and (not *dynamic-p*) (or (null pos) (> pos parent-end)))
                   (setf env (set-env-value env :end (get-env-value env :parent-end)))
                   (values parent-end env))
                  ((and pos (< pos *dynamic-pos*))
                   (cond ((< pos (1- parent-end))
                           ;; :form is the default
                           (setf end-type (or (get-env-value env :end-type) :form))
                           (ecase end-type
                              (:form  ; setf?
                               (setf env (set-env-value env :end (build-end pos))))
                              (:parent
                               (setf env (set-env-value env :end (get-env-value env :parent-end))))
                              (:subform
                               (setf env (set-env-value env :end (get-env-value env :subform-end))))))
                          (t
                           (setf env (set-env-value env :end (get-env-value env :parent-end))))) 
                   (values pos (set-env-value env :start pos)))
                  ((and (not (not *dynamic-p*)) (>= *dynamic-pos* parent-end))
                   (setf env (set-env-value env :end (get-env-value env :parent-end)))
                   (values parent-end env)))))))

(defun end-action (pos env &optional (dec-p nil))
   (when pos  
      (if dec-p 
        (values (1- (get-end env)) env)
        (values (get-end env) env))))

(defun parent-start-action (pos env)
   (when pos
      (values (buffer-bwd-sexp *buffer* (get-parent-end env)) env)))

(defun end-p (pos env)
   (when pos
      (let ((end (get-end env)))
        ;; unmatched quotes have no end!
        (when end
           (or (>= pos (1- end))
                (= *dynamic-pos* end))))))
               ;; (null (next-sexp-start pos)))))))

(defun inc-action (pos env)
   "Move past the left paren."
   (when pos
      (if (char= (buffer-char *buffer* pos) #\()
        (values (1+ pos) env)
        (values pos env))))

(defun get-char-action (pos env)
   (when pos
      (let ((char (buffer-char *buffer* pos)))
        (if (or (char= char #\() (whitespacep char))
          (if (< pos *dynamic-pos*) 
            (values (1+ pos) nil)
            nil)
          (values pos env)))))

(defun next-sexp-action (pos env)
   (when pos
      (setf pos (next-sexp-start pos))
      (when (and pos (< pos *dynamic-pos*))
         (values pos env))))

(defun unstyle-action (pos env)
   (when pos
      (let*((sexp-start (next-sexp-start (1+ pos)))
            (sexp-end (when sexp-start (sexp-end sexp-start))))
        (when (and sexp-end (not (char= (buffer-char *buffer* sexp-start) #\()))
           (style-region *generic-text-style* pos sexp-end))
        (values pos env))))

(defun parse-subform-action (pos env)
   (when pos
      (let ((end (get-end env)))
        (when end ; an unmatched quote or a preceding semicolon can cause end to be nil
           (let* ((sexp-start (if (char= (buffer-char *buffer* pos) #\() 
                                pos (next-sexp-start pos)))
                  (sexp-end (when sexp-start (buffer-fwd-sexp *buffer* sexp-start end)))
                  (current-char (when sexp-start (buffer-char *buffer* sexp-start))))
             ; (format t "~%sexp-start: ~s" sexp-start)
             ; (format t "~%sexp-end: ~s" sexp-end)
             ; (format t "~%char: ~s" current-char)
             (cond ((or (null sexp-start) (null sexp-end) (> sexp-start *dynamic-pos*)
                          (embedded-in-segment-p sexp-start))
                      (if *dynamic-p*
                        (cond ((and sexp-start (< sexp-start *dynamic-pos*) 
                                      (not (= sexp-start pos)))
                                 ; (format t "~%inner sexp-start: ~s" sexp-start)
                                 ; (format t "~%*dynamic-pos*: ~s" *dynamic-pos*)
                                 ; (break)
                                 (return-from parse-subform-action (values sexp-start nil)))
                                (t
                                 ; (format t "~%inner sexp-start: ~s" sexp-start)
                                 ; (format t "~%inner sexp-end: ~s" sexp-end)
                                 (return-from parse-subform-action nil)))
                        (return-from parse-subform-action (values (or sexp-end end) nil))))
                     ((> sexp-start end)
                      (if (not *dynamic-p*)
                        (return-from parse-subform-action (values (or sexp-end end) nil))
                        (cond ((> sexp-start *dynamic-pos*)
                                 (return-from parse-subform-action nil))
                                (t
                                 (return-from parse-subform-action (values end nil)))))))
             ; (format t "~%sexp-start2: ~s" sexp-start)
             ; (format t "~%sexp-end2: ~s" sexp-end)
             ; (format t "~%char2: ~s" current-char)
             (when (or (char= current-char #\')
                         (char= current-char #\`)
                         (char= current-char #\,))
                (incf sexp-start) 
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\@)
                (incf sexp-start) 
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\')
                (incf sexp-start) 
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\,)
                (incf sexp-start) 
                (setf current-char (buffer-char *buffer* sexp-start)))
             (cond ((char= current-char #\()
                      (when (and (>= *dynamic-pos* sexp-start))
                         (values sexp-start (set-env-value env :subform-start sexp-start))))
                     ((char= current-char #\#)
                      (cond ((and (< (+ sexp-start 2) (buffer-size *buffer*))
                                    (char= (buffer-char *buffer* (+ sexp-start 1)) #\')
                                    (char= (buffer-char *buffer* (+ sexp-start 2)) #\()
                                    (>= *dynamic-pos* sexp-start))
                               (values sexp-start (set-env-value env :subform-start (+ sexp-start 2))))
                              ((> *dynamic-pos* (+ sexp-start 2)) ; reader macro, skip
                               (values sexp-end nil))
                              (t nil)))
                     (t 
                      (if (= sexp-start pos)
                        (values (1+ pos) nil)
                        (values sexp-end nil)))))))))

(defun parse-loop-action (pos env)
   (when pos
      (let ((end (get-end env)))
        (when end
           (let* ((sexp-start (if (char= (buffer-char *buffer* pos) #\() 
                                pos (next-sexp-start pos)))
                  (sexp-end (when sexp-start (buffer-fwd-sexp *buffer* sexp-start end)))
                  (current-char (when sexp-start (buffer-char *buffer* sexp-start))))
             ; (format t "~%sexp-start: ~s" sexp-start)
             ; (format t "~%sexp-end: ~s" sexp-end)
             ; (format t "~%char: ~s" current-char)
             (cond ((or (null sexp-start) (null sexp-end) (> sexp-start *dynamic-pos*)
                          (embedded-in-segment-p sexp-start))
                      (if *dynamic-p*
                        (cond ((and sexp-start (< sexp-start *dynamic-pos*) 
                                      (not (= sexp-start pos)))
                                 ;; (format t "~%inner sexp-start: ~s" sexp-start)
                                 ;; (format t "~%*dynamic-pos*: ~S" *dynamic-pos*)
                                 ;; (break)
                                 (return-from parse-loop-action (values sexp-start nil)))
                                (t
                                 (return-from parse-loop-action nil)))
                        (return-from parse-loop-action (values (or sexp-end end) nil))))
                     ((> sexp-start end)
                      (if (not *dynamic-p*)
                        (return-from parse-loop-action (values (or sexp-end end) nil))
                        (cond ((> sexp-start *dynamic-pos*)
                                 (return-from parse-loop-action nil))
                                (t
                                 (return-from parse-loop-action (values end nil)))))))
             ;; (format t "~%sexp-start: ~S" sexp-start)
             ;; (format t "~%sexp-end: ~S" sexp-end)
             ;; (format t "~%char: ~S" current-char)
             (when (or (char= current-char #\')
                         (char= current-char #\`)
                         (char= current-char #\,))
                (incf sexp-start)
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\@)
                (incf sexp-start)
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\')
                (incf sexp-start)
                (setf current-char (buffer-char *buffer* sexp-start)))
             (when (char= current-char #\,)
                (incf sexp-start)
                (setf current-char (buffer-char *buffer* sexp-start)))
             (cond ((char= current-char #\()
                      (when (and (>= *dynamic-pos* sexp-start))
                         (values sexp-start (set-env-value env :subform-start sexp-start))))
                     ((char= current-char #\#)
                      (cond ((and (< sexp-start (1+ (buffer-size *buffer*)))
                                    (char= (buffer-char *buffer* (+ sexp-start 1)) #\')
                                    (char= (buffer-char *buffer* (+ sexp-start 2)) #\()
                                    (>= *dynamic-pos* sexp-start))
                               (values sexp-start (set-env-value env :subform-start (+ sexp-start 2))))
                              ((>= *dynamic-pos* (+ sexp-start 2))
                               (values sexp-end nil))
                              (t nil)))
                     ((alpha-char-p current-char) ; possibly a loop keyword
                      (values sexp-start (set-env-value env :subform-start sexp-start)))
                     (t
                      (if (= sexp-start pos)
                        (values (1+ pos) nil)
                        (values sexp-end nil)))))))))

(defun subform-action (pos env)
   (when pos 
      (let* ((subform-start (get-env-value env :subform-start))
             (subform-end (sexp-end subform-start)))
        (when (and (stack-empty-p) (char= (buffer-char *buffer* subform-start) #\())
           (let ((char-1 (buffer-char *buffer* (max (1- subform-start) 0))))
             (when (or (char-eolp char-1) (= subform-start 0))
                (setf (p-new-context-p *parser*) t)))) ; start of top-level form
        (if (and subform-end (>= pos subform-end))
          (values pos nil) ; ***
          (values subform-start env)))))

(defun default-value-start-action (pos env)
   (when pos
      (let ((char (buffer-char *buffer* pos)))
        (cond ((not (whitespacep char))
                ;; *** :subform 
                (values (get-env-value env :default-list-start) env))
               (t
                (when (< pos *dynamic-pos*)
                   :inc-pos))))))

(defun list-form-p (pos env)
   (declare (ignore env))
   (when pos
      (let ((start (next-sexp-start (1+ pos))))
        (when start (char= (buffer-char *buffer* start) #\()))))

(defun generic-form-p (pos env)
   (when pos
      (let* ((symbol-start (next-sexp-start (1+ pos)))
             (symbol-end (sexp-end symbol-start))
             (subform-end (sexp-end (get-env-value env :subform-start)))
             (char (when (and symbol-start subform-end (< symbol-start subform-end))
                      (buffer-char *buffer* symbol-start))))
        (when char
           (or (char= char #\()
                (char= char #\:)
                (char= char #\")
                (and symbol-end (> *dynamic-pos* symbol-end)))))))

(defun empty-form-p (pos env)
   (when pos
      (cond ((and (not *dynamic-p*) (null (next-sexp-start (1+ pos))))
              t)
             (t
              (let ((symbol-start (next-sexp-start (1+ pos)))
                    (subform-end (sexp-end (get-env-value env :subform-start))))
                (and symbol-start subform-end (>= symbol-start subform-end)))))))

(defun empty-loop-form-p (pos env)
   (when (and pos (not (alpha-char-p (buffer-char *buffer* pos))))
      (cond ((and (not *dynamic-p*) (null (next-sexp-start (1+ pos))))
              t)
             (t
              (let ((symbol-start (next-sexp-start (1+ pos)))
                    (subform-end (sexp-end (get-env-value env :subform-start))))
                (and symbol-start subform-end (>= symbol-start subform-end)))))))

(defun empty-form-action (pos env)
   (when pos
      (values (sexp-end pos) env)))
           
(defun defstyle-form-p (pos env)
   (when pos
      (let* ((symbol-start (next-sexp-start (1+ pos)))
             (symbol-end (sexp-end symbol-start))
             (string (when (and symbol-start symbol-end)
                        (buffer-substring *buffer* symbol-start symbol-end)))
             (defstyle-network (when string (get-defstyle-network *rtn-grammar* string))))
        (when defstyle-network
           (setf env (set-env-value env :defstyle-network defstyle-network)) t))))

;; *** redo of prev
(defun defstyle-form-action (pos env)
   (when pos
      (let* ((symbol-start (next-sexp-start (1+ pos)))
             (symbol-end (sexp-end symbol-start))
             (string (buffer-substring *buffer* symbol-start symbol-end))
             (styling-network (get-defstyle-network *rtn-grammar* string)))
        (when styling-network
           (values pos (set-env-value env :defstyle-network styling-network))))))

;;; actually package-or-keyword-p
(defun package-form-p (pos env)
   (declare (ignore env))
   (when (and pos
                 (or *do-cl-package* *do-keyword-package*))
      (let* ((symbol-start (next-sexp-start (1+ pos)))
             (symbol-end (sexp-end symbol-start)))
        (or (and *do-keyword-package* (char= (buffer-char *buffer* pos) #\:))
             (and *do-cl-package* symbol-start symbol-end
                   (find-symbol 
                    (string-upcase (buffer-substring *buffer* symbol-start symbol-end)) :cl))))))

;;; *** redo of the above
(defun package-form-action (pos env)
   (when (and pos (<= pos *dynamic-pos*)
                 (or *do-cl-package* *do-keyword-package*))
      (let* ((symbol-start (next-sexp-start (1+ pos)))
             (symbol-end (sexp-end symbol-start)))
        (cond ((and *do-keyword-package* (char= (buffer-char *buffer* pos) #\:))
                (style-region (keyword-package-style) symbol-start symbol-end) t)
               ((and *do-cl-package* 
                      ;; This test seems redundant, but a stack backup following a delete
                      ;; which mangles the symbol can cause probs.  Therefore check again.
                      (find-symbol 
                       (string-upcase (buffer-substring *buffer* symbol-start symbol-end)) :cl))
                (style-region (cl-package-style) symbol-start symbol-end)))
        (when (> *dynamic-pos* symbol-end)
           (values symbol-end env)))))

(defun parse-action (pos env)
   (when (and pos (<= pos *dynamic-pos*))
      (let ((form-end (get-end env)))
        (flet ((not-char-constant-p (element-start)
                 (or (< element-start 2)
                      (char/= (buffer-char *buffer* (- element-start 1)) #\\)
                      (char/= (buffer-char *buffer* (- element-start 2)) #\#)))
               (check-dynamic-p (element-start element-end)
                  (or (not *dynamic-p*)
                       (and (>= *dynamic-pos* element-start)
                             (<= *dynamic-pos* element-end)))))
          (do* ((element-start (if (char= (buffer-char *buffer* pos) #\()
                                  pos (next-sexp-start pos))
                                (when element-end (next-sexp-start element-end)))
                 (element-end (when element-start (sexp-end element-start))
                              (when element-start (sexp-end element-start)))
                 (current-char (when element-start (buffer-char *buffer* element-start))
                               (when element-start (buffer-char *buffer* element-start))))
                ((or (null element-start) (null element-end)
                      (when form-end (> element-start form-end))
                      (> element-start *dynamic-pos*))
                 ; (format t "~%fail: form-end: ~s" form-end)                 
                 ; (format t "~%fail: element-start: ~s" element-start)
                 ; (format t "~%fail: element-end: ~s" element-end)
                 ; (format t "~%fail: current-char: ~s" current-char)
                 (if *dynamic-p*
                   (if (and element-start (> element-start *dynamic-pos*))
                     (if (and form-end (<= form-end *dynamic-pos*))
                       (values form-end nil)
                       nil)
                     (if (and form-end (<= form-end *dynamic-pos*))
                       (values form-end nil)
                       (values pos nil)))
                   (if form-end
                     (values form-end nil)
                     (values pos nil))))
             ; (format t "~%form-end: ~s" form-end)                 
             ; (format t "~%element-start: ~s" element-start)
             ; (format t "~%element-end: ~s" element-end)
             ; (format t "~%current-char: ~s" current-char)
             (when (or (not *segment-array*)
                         (not-embedded-in-segment-p *segment-array* element-start))
                (when (or (char= current-char #\')
                            (char= current-char #\`)
                            (char= current-char #\,))
                   (incf element-start)
                   (setf current-char (buffer-char *buffer* element-start)))
                (when (char= current-char #\@)
                   (incf element-start)
                   (setf current-char (buffer-char *buffer* element-start)))
                (when (char= current-char #\')
                   (incf element-start)
                   (setf current-char (buffer-char *buffer* element-start)))
                (when (char= current-char #\,)
                   (incf element-start)
                   (setf current-char (buffer-char *buffer* element-start)))
                (cond ((and (char= current-char #\()
                              (not-char-constant-p element-start)
                              (check-dynamic-p element-start element-end))
                        (set-env-value env :current-char current-char)
                        (return-from parse-action (values element-start env)))
                       ((and (char= current-char #\#)
                              (< element-start (- (buffer-size *buffer*) 2))
                              (char= (buffer-char *buffer* (+ element-start 1)) #\')
                              (char= (buffer-char *buffer* (+ element-start 2)) #\()
                              (check-dynamic-p element-start element-end))
                        (set-env-value env :current-char current-char)
                        (return-from parse-action (values element-start env)))
                       ((and *do-keyword-package*
                              (char= current-char #\:) 
                              (not-char-constant-p element-start))
                        (set-env-value env :current-char current-char)
                        (return-from parse-action (values element-start env)))
                       ((and (char= current-char #\") (not-char-constant-p element-start))
                        (set-env-value env :current-char current-char)               
                        (return-from parse-action (values element-start env))))))))))

(defun char-action (pos env)
   (values pos (delete-env-value env :current-char)))

(defun macro-action (pos env accessor)
   (when (and pos (<= pos *dynamic-pos*)) ; *** ideom
      (let* ((symbol-start (next-sexp-start pos))
             (symbol-end (sexp-end symbol-start))
             (string (when (and symbol-start symbol-end)
                        (buffer-substring *buffer* symbol-start symbol-end)))
             (defstyle-network (get-defstyle-network *rtn-grammar* string)))
        ;; This seems redundant but there can be a stack-backup after a delete which 
        ;; mangles the macro name.  Therefore the check
        (when defstyle-network
           (style-region (funcall accessor) symbol-start symbol-end)
           (when (> *dynamic-pos* symbol-end) ; *** ideom
              (values symbol-end env))))))

(defun symbol-action (pos env accessor)
   (when (and pos (<= pos *dynamic-pos*))
      (let* ((symbol-style-accessor accessor)
             (start (next-sexp-start pos))
             (end (sexp-end start)))
        (unless end (return-from symbol-action nil))
        (cond (*do-exported-symbols*
                 (let ((name (string-upcase 
                              (buffer-substring *buffer* start end))))
                   (when name
                      (multiple-value-bind (symbol kind)
                                             (find-symbol name *current-package*)
                         (cond ((and symbol *current-package* (eq kind :external)
                                       (not (eq (funcall symbol-style-accessor)
                                                 (variable-definition-symbol-style))))
                                  (cond ((char= (buffer-char *buffer* start) #\")
                                           (style-region (exported-symbol-style)
                                                         start end nil))
                                          (t
                                           (style-region (exported-symbol-style)
                                                         start end))))                                  
                                 (t
                                  (cond ((char= (buffer-char *buffer* start) #\")
                                           (style-region (funcall symbol-style-accessor)
                                                         start end nil))
                                          (t
                                           (style-region (funcall symbol-style-accessor)
                                                         start end)))))))))
                (t
                 (cond ((char= (buffer-char *buffer* start) #\")
                          (style-region (funcall symbol-style-accessor) start end nil))
                         (t
                          
                          (cond (symbol-style-accessor
                                   (style-region (funcall symbol-style-accessor) start end))
                                  (t
                                   (format t "~%missing style accessor: ~s" symbol-style-accessor)
                                   (break)
                                   ))))))
        (when (> *dynamic-pos* end)
           (values end env)))))

;;; Same as above, but never style as an exported sym
(defun case-match-action (pos env accessor)
   (when (and pos (<= pos *dynamic-pos*))
      (let* ((symbol-style-accessor accessor)
             (start (if (whitespacep (buffer-char *buffer* pos)) (next-sexp-start pos) pos))
             (end (sexp-end start)))
        (unless end (return-from case-match-action nil))
        (cond ((char= (buffer-char *buffer* start) #\")
                 (style-region (funcall symbol-style-accessor) start end nil))
                (t
                 (cond (symbol-style-accessor
                          (style-region (funcall symbol-style-accessor) start end))
                         (t
                          (format t "~%missing style accessor: ~s" symbol-style-accessor)
                          (break)
                          ))))
        (when (> *dynamic-pos* end)
           (values end env)))))

(defun qualifier-action (pos env)
   (when pos
      (let* ((start (next-sexp-start pos))
             (end (when start (sexp-end start))))
        (when (and start end (< start *dynamic-pos*))
           (cond ((char= (buffer-char *buffer* start) #\:)
                    (style-region (keyword-package-style) start end)
                    (when (> *dynamic-pos* end)
                       (values end env)))
                   (t
                    (values pos env)))))))

(defun parameter-list-parse-action (pos env)
   (when (and pos (< pos *dynamic-pos*))
      (let ((char (buffer-char *buffer* pos))
            (option-p (get-env-value env :option-p)))
        (cond ((char= char #\&) 
                (set-env-value env :parameter :lambda-list-keyword)
                (set-env-value env :option-p t)
                (values pos env))
               ((char= char #\()
                (cond (option-p
                        (set-env-value env :default-list-start pos)
                        (set-env-value env :parameter :default-list))
                       (t ; a parameter and specializer list
                        (set-env-value env :parameter :specializer-list)))
                (values pos env))
               ((alphanumericp char)
                (set-env-value env :parameter :simple-parameter)
                (values pos env))
               (t
                (set-env-value env :parameter nil)
                (values (1+ pos) nil))))))

;;; *** consolidate these with a third arg
(defun default-p (pos env)
   (when pos (eq (get-env-value env :parameter) :default-list)))

(defun specializer-p (pos env)
   (when pos (eq (get-env-value env :parameter) :specializer-list)))

(defun lambda-list-keyword-p (pos env)
   (when pos (eq (get-env-value env :parameter) :lambda-list-keyword)))

(defun parameter-action (pos env accessor)
   (when pos
      (let ((end (sexp-end pos)))
        (when end
          (style-region (funcall accessor) pos end)
          (when (> *dynamic-pos* end) ; ideom
             (values end (set-env-value env :parameter nil)))))))

(defun superparen-action (pos env &optional (style (superparen-style)))
   (when pos
      (let ((end (get-end env)))
        (when end 
           (cond ((= pos end)
                    (unless (and *dynamic-p* (> *dynamic-pos* end))
                       (let* ((code-list (fcodes style))
                              (ff (first code-list))
                              (ms (second code-list)))
                         (buffer-set-font-codes *buffer* ff ms 
                                                (1- pos) end)))
                    (values end env))
                   ;; ***
                   ((or (char= (buffer-char *buffer* pos) #\))
                         (char= (buffer-char *buffer* pos) #\())
                    (unless (> (1+ pos) (buffer-size *buffer*))
                       (unless (and *dynamic-p* (> *dynamic-pos* end))
                          (let* ((code-list (fcodes style))
                                 (ff (first code-list))
                                 (ms (second code-list)))
                            (buffer-set-font-codes *buffer* ff ms 
                                                   pos (1+ pos))))
                       (values (1+ pos) env)))
                   (t
                    (when (< pos *dynamic-pos*)
                       :inc-pos)))))))

(defun eval-when-superparen-action (pos env)
   (superparen-action pos env (eval-when-superparen-style)))

(defun loop-superparen-action (pos env)
   (superparen-action pos env (loop-superparen-style)))
   
;;; quote-p ?
(defun doc-p (pos env)
   (declare (ignore env))
   (when pos (char= (buffer-char *buffer* pos) #\")))

(defun doc-action (pos env)
   (when pos
      (let ((end (get-end env)))
        (style-region (doc-style) (get-env-value env :start) end nil)
        (when (and end  (>= *dynamic-pos* end))
           (values end env)))))

(defun optional-paren-action (pos env)
   (when pos
      (when (or (char= (buffer-char *buffer* pos) #\()
                  (char= (buffer-char *buffer* pos) #\)))
         (setf pos (1+ pos)))
      (values pos env)))

(defun derivation-list-action (pos env)
   (when pos 
      (let ((end (sexp-end pos)))
        (style-region (defclass-derivation-style) (1+ pos) (1- end))
        (when  (>= *dynamic-pos* end)
           (values end env)))))

(defun options-doc-p (pos env)
   (when pos (get-env-value env :doc)))

(defun options-doc-action (pos env)
   (when pos
      (let* ((start (next-sexp-start (1+ (get-env-value env :doc))))
             (end (sexp-end start)))
        (when (and start end (< start *dynamic-pos*))
           (style-region (doc-style) start end nil))
        (when (and end (> *dynamic-pos* end))
           (values end  (delete-env-value env :doc))))))
   
(defun options-keyword-action (pos env)
   (when pos
      (let* ((keyword-start (next-sexp-start (1+ pos)))
             (keyword-end (sexp-end keyword-start)))
        (when (and keyword-start keyword-end)
           (style-region (keyword-package-style) keyword-start keyword-end nil)
           (when (string-equal (buffer-substring *buffer* keyword-start keyword-end)
                                 ":documentation")
              (set-env-value env :doc keyword-end))
           
           (when (string-equal (buffer-substring *buffer* keyword-start keyword-end)
                                 ":include")
              (set-env-value env :include keyword-end)))
        
        (when (and keyword-end (> *dynamic-pos* keyword-end))
           (values pos env)))))

;;; *** colon-p ?
(defun options-keyword-p (pos env)
   (declare (ignore env))
   (when pos (char= #\: (buffer-char *buffer* pos))))

(defun paren-action (pos env)
   (when pos
      (let ((next-sexp (next-sexp-start (1+ pos))))
        (if next-sexp 
          (setf pos next-sexp)
          (incf pos)))
      (values pos env)))

(defun slot-name-action (pos env)
   (when pos
      (let* ((start (next-sexp-start (1+ pos)))
             (end (sexp-end start)))
        (when (and start end)
           (style-region (defclass-slot-style) start end)
           (when (> *dynamic-pos* end)
              (values pos env))))))

(defun struct-field-list-p (pos env)
   (declare (ignore env))
   (when pos (char= #\( (buffer-char *buffer* pos))))

(defun struct-include-p (pos env)
   (when pos (get-env-value env :include)))

(defun struct-include-action (pos env)
   (when pos
      (let* ((start (next-sexp-start (1+ (get-env-value env :include))))
             (end (sexp-end start)))
        (when (and start end (< start *dynamic-pos*))
           (style-region (defstruct-ancestor-style) start end nil))
        (when (and end (> *dynamic-pos* end))
           (values end  (delete-env-value env :include))))))

(defun struct-field-action (pos env)
   (when pos
      (let ((end (sexp-end pos)))
        (when end
           (style-region (defstruct-field-style) pos end)
           (when (> *dynamic-pos* end) ;*** ideom
              (values end env))))))

(defun struct-fields-list-action (pos env)
   (when pos
      (let* ((name-start (next-sexp-start (1+ pos)))
             (name-end (sexp-end name-start))) ; *** test
        (when (and name-start name-end)
           (style-region (defstruct-field-style) name-start name-end)
           (when (> *dynamic-pos* name-end) ; *** ideom
              (values pos env))))))

(defun ancestor-action (pos env)
   (when pos
      (let* ((start (next-sexp-start (1+ pos))) ; skip paren
             (end (when start (sexp-end start)))
             (string (when (and start end) (buffer-substring *buffer* start end))))
        (when (and string (string-equal string ":include"))
           (style-region (keyword-package-style) start end)
           (let* ((next-start (next-sexp-start end))
                  (next-end (sexp-end next-start)))
             (when (and next-start next-end)
                (style-region (defstruct-ancestor-style) next-start next-end)
                (when (> *dynamic-pos* next-end) ; *** ideom
                   (values next-end env))))))))

(defun variable-definitions-list-p (pos env)
   (declare (ignore env))
   (when pos (char= #\( (buffer-char *buffer* pos)))) ; *** ideom

(defun variable-definitions-symbol-action (pos env)
   (when pos
      (let* ((start (next-sexp-start (1+ pos)))
             (end (sexp-end start)))
        (when (and start end)
           (style-region (variable-definition-symbol-style) start end nil)
           (when (> *dynamic-pos* end) ; *** ideom
              (values pos env))))))

(defun variable-list-action (pos env)
   (when pos
      (let ((list-end (sexp-end pos)))
        (when list-end
           (do* ((var-start (next-sexp-start (1+ pos)) (next-sexp-start (1+ var-end)))
                  (var-end (when var-start (sexp-end var-start))
                           (when var-start (sexp-end var-start))))
                 ((or (null var-start) (null var-end) (> var-start list-end)))
              (style-region (variable-definition-symbol-style) var-start var-end nil))
           (when (> *dynamic-pos* list-end)
              (values list-end env))))))

(defun region-p (pos env)
   (declare (ignore env))
   (when pos
      (or (char= (buffer-char *buffer* pos) #\()
           (and (char= (buffer-char *buffer* pos) #\')
                 (char= (buffer-char *buffer* (min (1+ pos) (buffer-size *buffer*))) #\()))))

;;; *** consolidate ???
(defun list-p (pos env)
   (declare (ignore env))
   (when pos (char= (buffer-char *buffer* pos) #\()))

(defun colon-p (pos env)
   (declare (ignore env))
   (when pos (char= (buffer-char *buffer* pos) #\:)))

;;; *** next-sexp-start?
(defun keyword-p (pos env)
   (declare (ignore env))
   (when pos (char= (buffer-char *buffer* pos) #\:)))

(defun loop-keyword-p (pos env)
   (declare (ignore env))
   (when pos
      (let* ((symbol-start (next-sexp-start pos))
             (symbol-end (sexp-end symbol-start))
             (string (when (and symbol-start symbol-end)
                        (buffer-substring *buffer* symbol-start symbol-end))))
        (when string
           (loop-keywd-p *rtn-grammar* string)))))

(defun loop-atom-p (pos env)
   (declare (ignore env))
   (and pos (not (char= (buffer-char *buffer* pos) #\())))

(defun loop-atom-action (pos env)
   (let ((end (sexp-end pos)))
     (when (and end (> *dynamic-pos* end))
        (values end env))))

(defun quote-p (pos env)
   (declare (ignore env))
   (char= (buffer-char *buffer* pos) #\"))

;;; *** consolidate
(defun env-lparen-p (pos env)
   (declare (ignore pos))
   (char= (get-env-value env :current-char) #\())

;;; ***
(defun env-lambda-p (pos env)
   (declare (ignore pos))
   (char= (get-env-value env :current-char) #\#))

(defun env-quote-p (pos env)
   (declare (ignore pos))
   (char= (get-env-value env :current-char) #\"))

(defun env-keyword-p (pos env)
   (declare (ignore pos))
   (char= (get-env-value env :current-char) #\:))

(defun keyword-action (pos env)
   (when pos
      (let ((end (sexp-end pos)))
        (when end
           (style-region (keyword-package-style) pos end)
           (when (> *dynamic-pos* end) ; *** ideom
              (values end env))))))

(defun loop-keyword-action (pos env)
   (when pos
      (let ((end (sexp-end pos)))
        (when end
           (style-region (loop-keywords-style) pos end)
           (when (> *dynamic-pos* end)
              (values end env))))))

(defun complete-action (pos env)
   (when pos
      (let ((end (sexp-end pos)))
        (when (and end (> *dynamic-pos* end))
           (values end env)))))

(defun break-action (pos env)
   (declare (ignore pos env))
   (break))

(defun quote-action (pos env)
   (when pos
      (let ((end (sexp-end pos)))
        (cond (end
                (style-region *string-style* pos end nil)
                (values end env))
               (t 
                ;; ***
                (style-region *string-style* pos *dynamic-pos*)
                nil)))))
      
