;;;-*- mode: lisp; package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-specials.lisp
;;;
;;;      copyright  2006 Glen Foy, all rights reserved,
;;;
;;;      This file contains most of the utility's special variables and
;;;      some miscellaneous functions, utilities and macros.
;;;
;;; ****************************************************************************

(defPackage color-coded (:use :cl :ccl) (:nicknames "CC"))
(in-package "CC")

;;; conditional compilation of debugging code:
;;; (pushnew :cc-debug *features*)

(defConstant *cc-version* "color-coded version 2.0b6 - released 7/32/9")

(defParameter *parser* nil "The RTN parser.")
(defParameter *buffer* nil "The target Fred buffer.")
(defParameter *dynamic-p* nil "Styling incrementally?")
(defParameter *dynamic-pos* nil "Fred buffer-position during an incremental parse.")
(defParameter *prefs-manager* nil)
(defParameter *current-package* nil)
(defParameter *lisp-file-p* nil "Try not to style the grocery list.")
(defParameter *rtn-batch-styling-p* nil "Use the RTN algorithm for batch styling?") 
(defParameter *rtn-incremental-styling-p* t "Use the RTN algorithm for incremental styling?") 
(defParameter *ff* nil)
(defParameter *ms* nil)

;;; Utilities:
(defun s-spec (var) (assert (symbolp var)) (get var :style-spec))
(defun f-spec (var) (assert (symbolp var)) (getf (s-spec var) :font))
(defun c-spec (var) (assert (symbolp var)) (getf (s-spec var) :case))
(defun default-style (var) (assert (symbolp var)) (get var :default-style))
(defun docc (var) (assert (symbolp var)) (get var :doc))
(defun sample-code (var) (assert (symbolp var)) (get var :sample-code))
(defun accessor (var) (assert (symbolp var)) (get var :accessor))

;;; (s-spec *sharp-comment-style*)

(defun f-pair (style-spec)
  (let ((font-spec (getf style-spec :font))
        (case-spec (getf style-spec :case)))
    (when (and font-spec case-spec)
      (multiple-value-bind (ff ms) (font-codes font-spec)
        (list (list ff ms) case-spec)))))

;;; These are the builtin parameters not created by calls to defstyle.
;;; The default values will be overriden by values stored in the prefs file.
;;; These defaults are only used if the prefs file doesn't exist or is corrupted.
(defParameter *defclass-slot-style* nil)
(defParameter *defclass-superclass-style* nil)
(defParameter *defstruct-field-style* nil)
(defParameter *defstruct-ancestor-style* nil)
(defParameter *defmethod-specializer-style* nil)
(defParameter *parameter-style* nil)
(defParameter *generic-def-style* nil)
(defParameter *generic-def-symbol-style* nil)
(defParameter *generic-text-style* nil)
(defParameter *super-paren-style* nil)
(defParameter *eval-when-super-paren-style* nil)
(defParameter *loop-super-paren-style* nil)
(defParameter *documentation-style* nil)
(defParameter *semi-colon-comment-style* nil)
(defParameter *sharp-comment-style* nil)
(defParameter *string-style* nil)
(defParameter *cl-function-style* nil)
(defParameter *keyword-style* nil)
(defParameter *exported-symbol-style* nil)
(defParameter *loop-keywords-style* nil)
;;; only the white color is relevant here
(defParameter *background-color* nil)
(defParameter *vanilla-styling* nil)

;;; PREFS is initialized to this list.
(defParameter *non-defstyle-prefs*
  '(*defclass-slot-style*
    *defclass-superclass-style*
    *defstruct-field-style*
    *defstruct-ancestor-style*
    *defmethod-specializer-style*
    *parameter-style*
    *generic-def-style*
    *generic-def-symbol-style*
    *generic-text-style*
    *super-paren-style*
    *eval-when-super-paren-style*
    *loop-super-paren-style*
    *documentation-style*
    *semi-colon-comment-style*
    *sharp-comment-style*
    *string-style*
    *cl-function-style*
    *keyword-style*
    *exported-symbol-style*
    *loop-keywords-style*
    *background-color*
    *vanilla-styling*))

;;; The default values will be overriden by values stored in the prefs file.
;;; These defaults are only used if the prefs file doesn't exist or is corrupted.
;;; This list is used by USE_DEFAULTS.
(defParameter *non-defstyle-pref-defaults*
   '(
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)        ; *defclass-slot-style*
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)            ; *defclass-superclass-style*
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)            ; *defstruct-field-style*
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)            ; *defstruct-ancestor-style*
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)            ; *defmethod-specializer-style*
     (:font ("monaco" 9 :italic (:color 0)) :case :down)                 ; *parameter-style*
     (:font ("monaco" 9 :plain (:color 0)) :case (:capitalize 3))         ; *generic-def-style*
     (:font ("monaco" 9 :underline (:color 1445511)) :case :down)        ; *generic-def-symbol-style*
     (:font ("monaco" 9 :plain (:color 0)) :case :down)                  ; *generic-text-style*
     (:font ("monaco" 10 :plain (:color 1445511)) :case :down)           ; *super-paren-style*
     (:font ("monaco" 10 :plain (:color 1445511)) :case :down)           ; *eval-when-super-paren-style*
     (:font ("monaco" 10 :plain (:color 1445511)) :case :down)           ; *loop-super-paren-style*
     (:font ("geneva" 9 :plain (:color 8390229)) :case :down)            ; *documentation-style*
     (:font ("monaco" 9 :italic (:color 295001)) :case :down)            ; *semi-colon-comment-style*
     (:font ("monaco" 9 :italic (:color 295001)) :case :down)            ; *sharp-comment-style*
     (:font ("monaco" 9 :plain (:color 7555117)) :case :down)            ; *string-style*
     (:font ("monaco" 9 :plain (:color 1445511)) :case :down)            ; *cl-function-style*
     (:font ("geneva" 10 :plain (:color 1445511)) :case :down)           ; *keyword-style*
     (:font ("monaco" 9 :bold :underline (:color 1445511)) :case :up)     ; *exported-symbol-style*
     (:font ("monaco" 9 :bold (:color 1445511)) :case :down)            ; *loop-keywords-style*
     ;; only the color is relevant for this one:
     (:font ("monaco" 9 :underline (:color 16777215)) :case :down)       ; *background-color*
     (:font ("monaco" 9 :plain (:color 0)) :case :down)))                ; *vanilla-styling*

(do* ((vars *non-defstyle-prefs* (rest vars))
       (var (first vars) (first vars))
       (values *non-defstyle-pref-defaults* (rest values))
       (value (first values) (first values)))
      ((null var))
   (setf (get var :style-spec) value)
   (set var (f-pair (s-spec var))))

;;; non-font-spec styling preference:
(defParameter *tab-key-styling* t)
(defParameter *do-dynamic-styling* t)
(defParameter *do-cl-package* t)
(defParameter *do-keyword-package* t)
(defParameter *do-exported-symbols* nil)
(defParameter *use-white-i-beam* nil)
(defParameter *use-black-i-beam* nil)
(defParameter *use-default-i-beam* t)

(defParameter *styling-preferences-defaults* 
   '(
     t       ; *tab-key-styling*
     t       ; *do-dynamic-styling*
     t       ; *do-cl-package*
     t       ; *do-keyword-package*
     nil     ; *do-exported-symbols*
     nil     ; *use-white-i-beam*
     nil     ; *use-black-i-beam*
     t))     ; *use-default-i-beam*

(defParameter *styling-preferences* 
  '(*tab-key-styling* *do-dynamic-styling* *do-cl-package* *do-keyword-package*
    *do-exported-symbols* *use-white-i-beam* *use-black-i-beam* *use-default-i-beam*))
(defParameter *generic-style-variables*
  '(*generic-def-style* *generic-def-symbol-style* *generic-text-style*))
(defParameter *cl-function-style-variables*
  '(*lambda-style* *flet-style* *labels-style*))

(defVar *save-default-i-beam* nil)
(defVar *save-default-italic-i-beam* nil)
(defVar *white-i-beam* nil)
(defVar *white-italic-i-beam* nil)
(defVar *black-i-beam* nil)
(defVar *black-italic-i-beam* nil)
(defConstant %black-i-beam-id% 128)
(defConstant %white-i-beam-id% 129)
(defConstant %black-italic-i-beam-id% 130)
(defConstant %white-italic-i-beam-id% 131)
(defParameter *segment-list* nil)
(defParameter *segment-array* nil)

;;; style accessors
;;; There are no defstyle calls for these; They are not the names of macros,
;;; so the style accessors are not automatically created.  Define them here.
(defun defstruct-field-style () (or *defstruct-field-style* *generic-text-style*))
(defun defstruct-ancestor-style () (or *defstruct-ancestor-style* *generic-text-style*))
(defun superparen-style () (or *super-paren-style* *generic-text-style*))
(defun eval-when-superparen-style ( ) (or *eval-when-super-paren-style* *generic-text-style*))
(defun exported-symbol-style () (or *exported-symbol-style* *generic-text-style*))
(defun loop-superparen-style ( ) (or *loop-super-paren-style* *generic-text-style*))
(defun loop-keywords-style ( ) (or *loop-keywords-style* *generic-text-style*))
(defun specializer-style () (or *defmethod-specializer-style* *generic-text-style*))
(defun parameter-style () (or *parameter-style* *generic-text-style*))
(defun defclass-derivation-style () (or *defclass-superclass-style* *generic-text-style*))
(defun defclass-slot-style () (or *defclass-slot-style* *generic-text-style*))
(defun parameter-style () (or *parameter-style* *generic-text-style*))
(defun specializer-style () (or *defmethod-specializer-style* *generic-text-style*))
(defun doc-style () (or *documentation-style* *generic-text-style*))
(defun cl-package-style () 
  (if (and *do-cl-package* *cl-function-style*) *cl-function-style* *generic-text-style*))
(defun keyword-package-style () 
  (if (and *do-keyword-package* *keyword-style*) *keyword-style* *generic-text-style*))

;;; ----------------------------------------------------------------------------
;;; sample code and doc for each style-item-variable is stored as a property.
;;; The ones below are for the parameters in *non-defstyle-prefs*.  Calls to
;;; defstyle supply the doc and sample-code for defstyle-prefs.
;;; ----------------------------------------------------------------------------
;;;
(setf (get '*defclass-slot-style* :sample-code)
      "(defclass bacon-lettuce-and-tomato (sandwich)
   ((bacon :initform 'hickory-smoked :accessor blt-bacon)
    (lettuce :initform 'romano :accessor blt-lettuce)
    (tomato :initform 'ripe :accessor  blt-tomato))
   (:documentation \"Bread is inherited from sandwich\"))")

(setf (get '*defclass-slot-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(setf (get '*defclass-superclass-style* :sample-code)
      "(defclass bacon-lettuce-and-tomato (sandwich)
   ((bacon :initform 'hickory-smoked :accessor blt-bacon)
    (lettuce :initform 'romano :accessor blt-lettuce)
    (tomato :initform 'ripe :accessor  blt-tomato))
   (:documentation \"Bread is inherited from sandwich\"))")

(setf (get '*defclass-superclass-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(setf (get '*embedded-function-symbol-style* :sample-code)
      "(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (flet ((add-ingredient (ingredient)
           \"Add ingredient and stir.\"
           (push ingredient pot)
           (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))") 

(setf (get '*embedded-function-symbol-style* :doc)
      " The style of the names of functions
 created with labels or flet.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 


(intern "PIZZA-TO-GO" :common-lisp-user)
(export '(common-lisp-user::pizza-to-go) :common-lisp-user)

(setf (get '*exported-symbol-style* :sample-code)
      "(defun pizza-to-go (peppers onions mushrooms cheese)
  \"Use extra cheese, then export.\"
   (let ((crust (roll-dough)))
     (push peppers crust)
     (push onions crust)
     (push mushrooms crust)
     (push cheese crust)
     (push cheese crust)
     (deliver-pizza (bake crust))))")

(setf (get '*exported-symbol-style* :doc)
      " The style of the names of symbols
 which are exported from a package.
You must first turn on this option
below.") 

(setf (get '*loop-keywords-style* :sample-code)
     "(defun name-clash (accessor-name superclasses env)
  (loop for superclass in superclasses 
        for sd = (find-sub-sd superclass env)
        thereis (and sd
                     (find accessor-name (sd-refnames sd)))))")

(setf (get '*loop-keywords-style* :doc)
" The style of loop keywords.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")


(setf (get '*defmethod-specializer-style* :sample-code)
      "(defmethod make-supper ((roast roast-main-course))
   \"Cook for 3 hours with veggies.\"
   (let ((pot nil))
     (push roast pot)
     (push 'onions pot)
     (push 'carrots pot)
     (slow-cook pot 3)))")

(setf (get '*defmethod-specializer-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(setf (get '*defstruct-ancestor-style* :sample-code)
      "(defstruct (italian-meal (:include desert))
   \"Garlic is good for you.\"
   (appetizer 'calimari)
   (main-course 'veal-parmesan)
   salad)")

(setf (get '*defstruct-ancestor-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 

(setf (get '*defstruct-field-style* :sample-code)
      "(defstruct italian-meal
   \"Garlic is good for you.\"
   appetizer
   (pasta 'fettuchine)
   (main-course 'veal-parmesan)
   salad)")

(setf (get '*defstruct-field-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 

(setf (get '*generic-def-style* :sample-code)
      "(defxxxxx make-supper ((roast roast-main-course))
   \"Cook for 3 hours with veggies.\"
   (let ((pot nil))
     (push roast pot)
     (push 'onions pot)
     (push 'carrots pot)
     (slow-cook pot 3)))")

(setf (get '*generic-def-style* :doc)
      " This is the generic style given to
defxxxx forms when you click the 
use generic button.")    

(setf (get '*generic-def-symbol-style* :sample-code)
      "(defxxxxx make-supper ((roast roast-main-course))
   \"Cook for 3 hours with veggies.\"
   (let ((pot nil))
     (push roast pot)
     (push 'onions pot)
     (push 'carrots pot)
     (slow-cook pot 3)))")

(setf (get '*generic-def-symbol-style* :doc)
      " This is the generic style given to
defxxxx symbols when you click the
use generic button.")    

(setf (get '*super-paren-style* :sample-code)
      "(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (flet ((add-ingredient (ingredient)
           \"Add ingredient and stir.\"
           (push ingredient pot)
           (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))")

(setf (get '*super-paren-style* :doc)
      " Super-parens are the outermost 
parentheses of defxxx forms and
embedded function forms, created
with labels or flet.") 

(setf (get '*eval-when-super-paren-style* :sample-code)
      "(eval-when (:hungry :starved :in-need-of-comfort-food)
         (dotimes (count 3)
            (eat-supper))
       )")

(setf (get '*eval-when-super-paren-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(setf (get '*loop-super-paren-style* :sample-code)
     "(defun name-clash (accessor-name superclasses env)
  (loop for superclass in superclasses 
        for sd = (find-sub-sd superclass env)
        thereis (and sd
                     (find accessor-name (sd-refnames sd)))))")

(setf (get '*loop-super-paren-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(setf (get '*documentation-style* :sample-code)
      "(defclass peanut-butter-and-jelly (sandwich)
   ((peanut-butter :initform 'skippy :accessor pbj-peanut-butter)
    (jelly :initform 'grape :accessor pbj-jelly))
   (:documentation \"Bread is inherited from sandwich\"))")

(setf (get '*documentation-style* :doc)
      " The style of the documentation
of the definition forms.")    

(setf (get '*parameter-style* :sample-code)
      "(defmethod make-pasta ((fettuccine pasta) &optional (veggie 'broccolli) 
                                                       (sauce 'white-sauce))
   \"Don't over-cook the white sauce.\"
   (let ((plate nil))
      (push (cook-al-dente fettuccine) plate)
      (push (lightly-steamed veggie) plate)
      (pour sauce plate)))")

(setf (get '*parameter-style* :doc)
      " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")  

(setf (get '*string-style* :sample-code)
      "\"Beware the Jabberwock, my son
   The jaws that bite, the claws that catch!
Beware the Jubjub bird, and shun
   The frumious Bandersnatch!\"")

(setf (get '*string-style* :doc)
      " The style of strings.  Note: string 
style and documentation style are
distinct.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 

(setf (get '*semi-colon-comment-style* :sample-code)
      "(defclass peanut-butter-and-jelly (sandwich)
   ;; it's best to use crunchy, and slices without holes.
   ((peanut-butter :initform 'skippy :accessor pbj-peanut-butter)
    (jelly :initform 'grape :accessor pbj-jelly))
   (:documentation \"bread is inherited from sandwich\"))")

(setf (get '*semi-colon-comment-style* :doc)
      " Semi-colon comment style.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 

(setf (get '*sharp-comment-style* :sample-code)
      "#| 
Twas brillig, and the slithy toves
   Did gyre and gimble in the wabe;
all mimsy were the borogroves,
   and the mome raths outgrabe.
|#")

(setf (get '*sharp-comment-style* :doc)
      " Sharp-stroke comment style.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.") 

(setf (get '*cl-function-style* :sample-code)
      "(defun print-menu (status)
   (if (eql status 'friends)
     (format t \"~&Fillet Mignon.\")
     (format t \"~&pb&j\")))")

(setf (get '*cl-function-style* :doc)
      " This is the generic style given all 
functions, macros, and special forms 
in the :cl package.  You must first 
turn on this option below.") 

(setf (get '*keyword-style* :sample-code)
      "(defun slice-bread (&optional (loaf \"pumpernickle\"))
   \"Character-sized slices\"
   (remove-if 'has-holes-p 
              (coerce loaf 'list) :from-end t))")

(setf (get '*keyword-style* :doc)
      " This is the generic style given all 
the symbols in the :keyword package.  
You must first turn on this option 
below.  It is also the style given 
to lambda-list-keywords.")

(setf (get '*generic-text-style* :sample-code)
      "Beware the Jabberwock, my son
   The jaws that bite, the claws that catch!
Beware the Jubjub bird, and shun
   The frumious Bandersnatch!")

(setf (get '*generic-text-style* :doc)
      " This is the style of any text that 
does not have its own style variable. 
It can also be assigned to non-def
style variables by clicking the 
generic button.")

(setf (get '*background-color* :sample-code)
      "(defclass peanut-butter-and-jelly (sandwich)
   ((peanut-butter :initform 'skippy :accessor pbj-peanut-butter)
    (jelly :initform 'grape :accessor pbj-jelly))
   (:documentation \"Bread is inherited from sandwich\"))")

(setf (get '*background-color* :doc)
      " Background color info is not stored 
with a file.  All currently open Fred 
windows will receive the same color.")

(setf (get '*vanilla-styling* :sample-code)
      "Beware the Jabberwock, my son
   The jaws that bite, the claws that catch!
Beware the Jubjub bird, and shun
   The frumious Bandersnatch!")

(setf (get '*vanilla-styling* :doc)
      " This is the style used by the 
Vanilla Styling command in the 
Edit menu.")

;;; ----------------------------------------------------------------------------
;;; miscellaneous functions, utilities and macros
;;;
(defun clobber-check (functions)
   (let* ((function (first functions))
          (name (first function))
          (qualifier (first (second function)))
          (specializers (third function))
          function-list)
     (when function
        (cond (qualifier
                (setf function-list (cons name (cons qualifier (list specializers)))))
               (t ;no qualifiers
                (setf function-list (cons name (list specializers)))))
        
        (when (apply #'ccl::find-method-by-names  function)
           (message-dialog (format nil "Loading color-coded will clobber this method:

~S." function-list)
                           :size #@(550 140) 
                           :position (get-centered-dialog-position 510 230)))
        (clobber-check (rest functions)))))

;;; are we stepping on previously defined methods?
(unless (member "COLOR-CODED" *modules* :test 'string-equal)
   (clobber-check 
    '((cl:initialize-instance  (:around)  (ccl:scrolling-fred-view  t))
      (ccl:view-activate-event-handler (:before) (ccl:scrolling-fred-view))
      (ccl:view-activate-event-handler (:after) (ccl:window-fred-item))
      (ccl::window-make-parts (:after)  (ccl:fred-window  t))
      (ccl:view-deactivate-event-handler (:before) (ccl:scrolling-fred-view))
      (ccl:view-deactivate-event-handler (:around) (ccl::fred-v-scroll-bar))
      (ccl:view-deactivate-event-handler (:around) (ccl::fred-h-scroll-bar))
      (ccl:view-key-event-handler (:around) (ccl:window-fred-item t))
      (ccl:ed-indent-for-lisp (:before) (ccl:window-fred-item t t))
      (ccl:paste (:after) (ccl:window-fred-item))
      (ccl:ed-yank (:after) (ccl:window-fred-item)))))

(defMacro while (test &body body) `(do () ((not ,test)) ,@body))

;;; an environment is an alist of values used at each stack level:
(defMacro get-env-value (env name) `(second (assoc ,name ,env)))

(defMacro set-env-value (env name value)
   `(let ((existing-entry (assoc ,name ,env)))
      (cond (existing-entry
              (setf (second existing-entry) ,value))
             (t
              (push (list ,name ,value) ,env)))
      ,env))

;;; *** be sure to use this with a setf
(defMacro delete-env-value (env name)
   `(delete-if #'(lambda (entry)
                   (eq (first entry) ,name))
               ,env))

(defMacro set-previous-dynamic-pos (val)
   `(setf (p-previous-dynamic-pos *parser*) ,val))

(defMacro set-batch-processing () `(setf *dynamic-pos* most-positive-fixnum
                                             (p-previous-dynamic-pos *parser*) nil))

(defMacro sexp-end (sexp-start)
   "Returns the ending position of the sexp that starts at SEXP-START." 
   `(when ,sexp-start 
       (let ((end (buffer-fwd-sexp *buffer* ,sexp-start)))
         (if end
           end
           ;;; char constants seem not to work right with buffer-fwd-sexp, if pos
           ;;; is at the char, instead of the #
           (when (and (>= ,sexp-start 2)
                         (> (buffer-size *buffer*) (1+ ,sexp-start))
                         (char= (buffer-char *buffer* (- ,sexp-start 1)) #\\)
                         (char= (buffer-char *buffer* (- ,sexp-start 2)) #\#))
              (1+ ,sexp-start))))))


(defMacro sexp-start (pos)
  "Returns the starting position of the sexp that ends at POS." 
  `(when ,pos (buffer-bwd-sexp *buffer* ,pos)))

;;; When styling dynamically, a region's end position can grow!!
;;; Call a closure to get the current end.
(defMacro get-end (env) `(let ((ufo (get-env-value ,env :end)))
                              (when ufo
                                ;; ***
                                (if (integerp ufo) ufo (funcall ufo)))))

(defMacro get-start (env) `(let ((ufo (get-env-value ,env :start)))
                               (when ufo
                                 (if (integerp ufo) ufo (funcall ufo)))))

(defMacro get-parent-end (env) `(let ((ufo (get-env-value ,env :parent-end)))
                                      (when ufo
                                        ;; ***
                                        (if (integerp ufo) ufo (funcall ufo)))))

(defMacro get-parent-start (env) `(let ((ufo (get-env-value ,env :parent-start)))
                                      (declare (ignore ufo))
                                      0))

(defMacro get-subform-end (env) `(let ((start (get-env-value ,env :subform-start)))
                                        (when start (sexp-end start)))) 

(defMacro get-current-char (env) `(get-env-value ,env :current-char))


(defMacro build-end (pos) `#'(lambda () (sexp-end ,pos)))

(defMacro atom-start (start)
   `(when ,start
       (do* ((pos ,start (1- pos))
              (char (when (>= pos 0) (buffer-char *buffer* pos))
                    (when (>= pos 0) (buffer-char *buffer* pos))))
             ((or (null char)
                   (whitespacep char) (char= char #\() 
                   (char= char #\)) (char= char #\")) (1+ pos)))))

#|
(defun atom-start (start)
   (when start
      (do* ((pos start (1- pos))
             (char (when (>= pos 0) (buffer-char *buffer* pos))
                   (when (>= pos 0) (buffer-char *buffer* pos))))
            ((or (null char)
                  (whitespacep char) (char= char #\() 
                  (char= char #\)) (char= char #\")) (1+ pos)))))
|#

(defMacro atom-end (start)
   `(when ,start
       (let ((buffer-size (buffer-size *buffer*)))
         (do* ((pos ,start (1+ pos))
                (char (when (<= pos buffer-size) (buffer-char *buffer* pos))
                      (when (<= pos buffer-size) (buffer-char *buffer* pos))))
               ((or (null char)
                     (whitespacep char) (char= char #\)) (char= char #\() 
                     (char= char #\") (char= char #\;)) pos)))))

#|
(defun atom-end (start)
   (when start
      (let ((buffer-size (buffer-size *buffer*)))
        (do* ((pos start (1+ pos))
               (char (when (<= pos buffer-size) (buffer-char *buffer* pos))
                     (when (<= pos buffer-size) (buffer-char *buffer* pos))))
              ((or (null char)
                    (whitespacep char) (char= char #\)) (char= char #\() 
                    (char= char #\") (char= char #\;)) pos)))))
|#

;;; Quoted lists at the end of a form return nil:
;;; (defParameter p '((:w . 0)))
;;; pos:             ^
;;; buffer-bwd-sexp
(defMacro next-sexp-start (pos)
  `(when ,pos
     (do* ((position ,pos)
           (start (buffer-bwd-sexp *buffer* (buffer-fwd-sexp *buffer* position nil nil t))
                  (buffer-bwd-sexp *buffer* (buffer-fwd-sexp *buffer* position nil nil t))))
          ((or (null start) (>= start position)) start)
       (setf position (buffer-fwd-sexp *buffer* position nil nil t))
       (when (null position) (return nil)))))
       
;;; ----------------------------------------------------------------------------
;;; These six functions are borrowed from JAA's restyled-definitions.lisp.  In
;;; fact, the whole capitalization mechanism comes from restyled-definitions.lisp.
;;;
(defMethod buffer-set-case ((buffer buffer-mark) (case (eql :down)) start end)
  (buffer-downcase-region buffer start (min (buffer-size buffer) end)))

(defMethod buffer-set-case ((buffer buffer-mark) (case (eql :up)) start end)
  ;; don't use eupcase region...
  (buffer-upcase-symbol buffer start (min (buffer-size buffer) end)))

(defMethod buffer-set-case ((buffer buffer-mark) (case (eql :unchanged)) start end)
  (declare (ignore start end))
  ())

(defMethod buffer-set-case ((buffer buffer-mark) (case (eql :cap)) start end)
  (buffer-capitalize-region buffer start (min (buffer-size buffer) end)))

(defMethod buffer-set-case ((buffer buffer-mark) (case (eql :cap3)) start end)
  (buffer-set-case buffer :down start end)
  (buffer-capitalize-region buffer (+ start 3) (+ start 4)))

;;; This is for compatibility with an older prefs file format:
(defMethod buffer-set-case ((buffer buffer-mark) (case cons) start end)
   (buffer-set-case buffer :cap3 start end))

(defMethod buffer-upcase-symbol ((buffer buffer-mark) start end)
   ;; upcases all nonescaped characters in region
   (let (char escaped)
     (when (> end start)
        (do ((pos start (1+ pos)))
             ((>= pos end))
           (setf char (buffer-char buffer pos))
           (cond (; (char= char #\\) (incf pos))
                    (char= char #\|) (setf escaped (not escaped)))
                   ((and (lower-case-p char) (not escaped))
                    (buffer-char-replace buffer (char-upcase char) pos)))))))

(defMacro fcodes (var) `(first ,var))
(defMacro fcase (var) `(second ,var))

(defMacro style-region (style start end  &optional (set-caps-p t))
   "This is the basic styling macro that calls BUFFER-SET-FONT-CODES and BUFFER-SET-CASE."
   (cond (set-caps-p
           `(progn
               (when (or (not *dynamic-p*)
                           (and *dynamic-p* 
                                 (>= *dynamic-pos* ,start)
                                 (<= *dynamic-pos* (1+ ,end))))
                  (let* ((code-list (fcodes ,style))
                         (ff (first code-list))
                         (ms (second code-list)))
                    (buffer-set-font-codes *buffer* ff ms ,start ,end)
                    (buffer-set-case *buffer*
                                     (fcase ,style)
                                     ,start ,end)))))
          (t
           `(progn
               (when (or (not *dynamic-p*)
                           (and *dynamic-p* 
                                 (>= *dynamic-pos* ,start)
                                 (<= *dynamic-pos* (1+ ,end))))
                  (let* ((code-list (fcodes ,style))
                         (ff (first code-list))
                         (ms (second code-list)))
                    (buffer-set-font-codes *buffer* ff ms ,start ,end)))))))

#+rmcl
(defun char-eolp (char)
  (memq char `(#\return #\linefeed ,(code-char #x2028) ,(code-char #x2029))))

(defun background-color ()
   "Get the color value from the style-item-variable spec."
   (second (first (last (f-spec '*background-color*)))))

(defun current-package ()
   (let ((fred (find-if #'(lambda (w)
                           ;; *** typep
                           (not (string-equal (window-title w) "listener")))
                        (windows :class 'fred-window))))
     (when fred 
        (ignore-errors ; in case the package is not defined
         (window-package fred)))))

(defun get-centered-dialog-position (dialog-width dialog-height )
   "Calculate the centered dialog screen position."
   (let* ((screen-center-width-coord (/ *screen-width* 2))
          (screen-center-height-coord (/ *screen-height* 2))
          (dialog-x-coord (round (- screen-center-width-coord (/ dialog-width 2))))
          (dialog-y-coord (round (- screen-center-height-coord (/ dialog-height 2)))))
     (make-point dialog-x-coord dialog-y-coord)))

(defMethod set-background-color ((fred fred-item)) 
   (set-part-color fred :body (background-color)))

(defMethod set-generic-text-style (buffer &optional (start 0) (end (buffer-size buffer)))
  (buffer-set-font-spec buffer (f-spec '*generic-text-style*) start end))

;;; This function is stolen from the Interface Toolkit.
(defun make-i-beam (data-string mask-string hotspot)
   (when (or (> (length (string data-string)) 64)
               (> (length (string mask-string)) 64))
      (error "data-string & mask-string must be < 64 chars long"))
   (rlet ((data :bits16)
          (mask :bits16))
     (with-pstrs ((data-str data-string)
                  (mask-str mask-string))
       (#_stuffhex :ptr data :ptr data-str)
       (#_stuffhex :ptr mask :ptr mask-str))
     (make-record :cursor
                  :data data
                  :mask mask
                  :hotspot hotspot)))

(defun make-black-i-beam ()
  (make-i-beam "0C60028001000100010001000100010001000100010001000100010002800C60"
               "0C60028001000100010001000100010001000100010001000100010002800C60"
               #@(4 7)))

(defun make-white-i-beam ()
   (make-i-beam "0000000000000000000000000000000000000000000000000000000000000000"
                "0C60028001000100010001000100010001000100010001000100010002800C60"
                #@(4 7)))

(defun make-italic-black-i-beam ()
  (make-i-beam "031800A0004000400080008001000100010002000200040004000A0031800000"
               "031800A0004000400080008001000100010002000200040004000A0031800000"
               #@(7 7)))

(defun make-italic-white-i-beam ()
   (make-i-beam "0000000000000000000000000000000000000000000000000000000000000000"
                "031800A0004000400080008001000100010002000200040004000A0031800000"
                #@(7 7)))

(defun init-i-beams ()
   (setf *save-default-i-beam* ccl:*i-beam-cursor*)
   (setf *save-default-italic-i-beam* ccl::*italic-i-beam-cursor*)
   (setf *black-i-beam* (make-black-i-beam))
   (setf *white-i-beam* (make-white-i-beam))
   (setf *black-italic-i-beam* (make-italic-black-i-beam))
   (setf *white-italic-i-beam* (make-italic-white-i-beam)))

(init-i-beams)

(defun set-white-i-beam ()
  (setf ccl::*i-beam-cursor* *white-i-beam*
        ccl::*italic-i-beam-cursor* *white-italic-i-beam*))

(defun set-black-i-beam ()
  (setf ccl::*i-beam-cursor* *black-i-beam*
        ccl::*italic-i-beam-cursor* *black-italic-i-beam*))

(defun set-default-i-beam ()
  (setf ccl::*i-beam-cursor* *save-default-i-beam*
        ccl::*italic-i-beam-cursor* *save-default-italic-i-beam*))

;;; These two methods are modifications of code from Shannon Spires's 
;;; window-background-colors.lisp.  They give the scroll-bar areas the same
;;; color as the background when the window is deacitvated.
(defMethod ccl:view-deactivate-event-handler :around ((bar ccl::fred-v-scroll-bar))
  (let ((*white-color* (background-color)))
    (call-next-method)))

(defMethod ccl:view-deactivate-event-handler :around ((bar ccl::fred-h-scroll-bar))
  (let ((*white-color* (background-color)))
    (call-next-method)))

;;; Keep the background of the doc, info, fred help, and trace windows white, and the 
;;; i-beam default.  A dark background can make the text disappear ...
;;;  Otherwise, set the i-beam accordingly.
(defMethod ccl:view-activate-event-handler :before ((view scrolling-fred-view))
  (let ((window (view-window view)))
    (cond ((or (typep window 'ccl::doc-output-class)
               (typep window 'ccl::trace-dialog)
               (typep window 'ccl::edit-anything-dialog)
               (typep window 'ccl::ed-help-window))
           (when (neq (part-color view :body) *white-color*)
             (set-part-color view :body *white-color*))
           (set-default-i-beam))
          (t 
           (cond (*use-white-i-beam* (set-white-i-beam))
                 (*use-black-i-beam* (set-black-i-beam))
                 (*use-default-i-beam* (set-default-i-beam)))))))

(defMethod ccl:view-deactivate-event-handler :before ((view scrolling-fred-view))
  (set-default-i-beam))

;;; Support for background coloring of the poof button.  This is a modification
;;; of code from Shannon Spires's window-background.lisp.
(defMethod ccl::window-make-parts :after ((w fred-window) &rest initargs)
   (declare (ignore initargs))
   (unless (or (typep w 'listener) (typep w 'ccl::doc-output-class)
                (typep w 'ccl::initargs-window) (ccl::window-buffer-read-only-p (window-key-handler w)))
      (set-part-color w :content (background-color))
      (set-part-color w :body (background-color))
      ;; font color needs to be compatible with the background for new windows.  
      (let ((spec (list (list :color (second (first (last (f-spec '*generic-text-style*))))))))
        (set-view-font w spec))))


;;; Clean up the Mac Heap
(defun thanks-a-heap ()
  "Clean up the Mac heap and set pointers to nil."
  ;; (dispose-record *black-i-beam*)
  (setf *black-i-beam* nil)
  ;; (dispose-record *white-i-beam*)
  (setf *white-i-beam* nil)
  ;; (dispose-record *black-italic-i-beam*)
  (setf *black-italic-i-beam* nil)
  ;; (dispose-record *white-italic-i-beam*)
  (setf *white-italic-i-beam* nil)
  (when *prefs-manager*
    (setf (styling-dialog *prefs-manager*) nil)))

;;; Before saving an image, clean up.
(pushnew 'thanks-a-heap ccl:*save-exit-functions*)
;;; When launching a saved image do this.
(pushnew 'init-i-beams ccl:*lisp-startup-functions*)
(pushnew 'cc::read-prefs-on-startup ccl:*lisp-startup-functions*)

