;;;-*- Mode: Lisp; Package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-rtn.lisp
;;;
;;;      copyright  2008 Glen Foy, all rights reserved,
;;;
;;;      A Data-Driven Recursive Transition Network Parser for incremental styling.
;;;
;;;      For a discussion of RTNs see Allen's _Natural Language Understanding_
;;;      For a discussion of the data-driven paradigm see Norvig's _PAIP_
;;;
;;; ****************************************************************************

(in-package "CC")

;;; ----------------------------------------------------------------------------
;;; Data-driven representation of the RTN grammar
;;; ----------------------------------------------------------------------------
;;;
;;; CONVENTIONS:
;;; Start node comes first in the node list.
;;; Nodes must have unique names within the entire grammar.
;;; Networks must have unique names within the entire grammar.
;;;
;;; The DEFSTYLE macro calls specify additional networks which are created 
;;; dynamically in construct-network below.  See color-coded-defstyle.lisp.

(defParameter *rtn-grammar-description*
   '((region
      ;;; A region is a portion of the buffer, possibly the entire buffer,
      ;;; that contains one or more forms. Parse-subform-action identifies 
      ;;; the next form to be styled. The network loops until all the forms 
      ;;; in the region have been styled.
      ;;;
      ;;; There are 3 types of forms: defstyle forms, cl package forms and generic forms.
      ;;;
      ;;; The network uses indirect recursion to parse forms of arbitrary complexity.
      ;;;
      ((rg1 ((act start-action) (jmp? end-p -> rg6 0)
             (jmp-fail parse-subform-action -> rg1 1)
             (jmp-fail subform-action -> rg1 2) (act unstyle-action) 
             (jmp? defstyle-form-p -> rg2 0) (jmp? package-form-p -> rg3 0)
             (jmp? generic-form-p -> rg4 0) (jmp? empty-form-p -> rg5 0) (jmp-stop -> rg1 3)))
       (rg2 ((push defstyle-form -> rg1 1) (jmp -> rg1 3)))
       (rg3 ((push package-form -> rg1 1) (jmp -> rg1 3)))
       (rg4 ((push generic-form -> rg1 1) (jmp -> rg1 3)))
       (rg5 ((act empty-form-action) (jmp -> rg1 1)))
       (rg6 ((pop)))))

     (defstyle-form
       ;;; Defstyle-forms are specified by the defstyle macro. The networks that
       ;;; parse the individual defstyle-forms are created by construct-network below.
       ;;; This network pushes the appropriate constructed network which actually 
       ;;; does the styling.
       ;;;
       ;;; See color-coded-defstyle.lisp and defstyle-form-action.
       ;;;
       ((ds1 ((act start-action) (act defstyle-form-action) (push defstyle-network -> ds2 0) (pop-fail)))
        (ds2 ((pop)))))

     (package-form
      ;;; Package-forms are forms that begin with a cl function or macro.
      ;;;
      ((pf1 ((act start-action) (pop-fail package-form-action)
             (jmp? end-p -> pf5 0) (jmp-fail parse-action -> pf1 2) 
             (jmp? end-p -> pf5 0) (jmp? env-lparen-p -> pf2 0) (jmp? env-lambda-p -> pf2 0)
             (jmp? env-keyword-p -> pf3 0) (jmp? env-quote-p -> pf4 0)))
       (pf2 ((act char-action) (push region -> pf1 2)))
       (pf3 ((act char-action) (act keyword-action) (jmp -> pf1 2)))
       (pf4 ((act char-action) (act quote-action) (jmp -> pf1 2)))
       (pf5 ((pop)))))

     (generic-form
      ;;; Generic-forms are forms which are not defstyle-forms or package-forms.
      ;;; Note that there are recursive calls to region in the three types of forms.
      ;;;
      ((gf1 ((act start-action) (act inc-action) 
             (jmp? keyword-p -> gf4 1) (jmp? quote-p -> gf5 1)
             (pop-fail parse-action) (jmp -> gf2 2)))
       (gf2 ((jmp? end-p -> gf6 0) (jmp-fail parse-action -> gf2 0)
             (jmp? end-p -> gf6 0) (jmp? env-lparen-p -> gf3 0) (jmp? env-lambda-p -> gf3 0) 
             (jmp? env-keyword-p -> gf4 0) (jmp? env-quote-p -> gf5 0)))
       (gf3 ((act char-action) (push region -> gf2 0)))
       (gf4 ((act char-action) (act keyword-action) (jmp -> gf2 0)))
       (gf5 ((act char-action) (act quote-action) (jmp -> gf2 0)))
       (gf6 ((pop)))))

     (generic-rest
      ;;; This is similar to generic-form except that the first
      ;;; symbol is not unstyled.
      ;;;
      ;;; This is used by slot-list, options, variable-definitions, etc. where 
      ;;; the first symbol receives a unique style that must be preserved, 
      ;;; and the rest of the form is processed generically.
      ;;;
      ((gr1 ((act start-action) (act inc-action) (jmp? end-p -> gr5 0) 
             (jmp-fail parse-action -> gr1 2) (jmp? end-p -> gr5 0)
             (jmp? env-lparen-p -> gr2 0) (jmp? env-lambda-p -> gr2 0) 
             (jmp? env-keyword-p -> gr3 0) (jmp? env-quote-p -> gr4 0)))
       (gr2 ((act char-action) (push region -> gr1 2)))
       (gr3 ((act char-action) (act keyword-action) (jmp -> gr1 2)))
       (gr4 ((act char-action) (act quote-action) (jmp -> gr1 2)))
       (gr5 ((pop)))))

     (parameter-list
      ((pl1 ((act start-action) (act inc-action) (jmp? end-p -> pl5 0)
             (jmp-fail parameter-list-parse-action -> pl1 2) (jmp? default-p -> pl2 0)
             (jmp? specializer-p -> pl3 0) (jmp? lambda-list-keyword-p -> pl4 0)
             (act parameter-action parameter-style) (jmp -> pl1 2)))
       (pl2 ((act inc-action) (act parameter-action parameter-style) 
             (act default-value-start-action) (push generic-rest -> pl1 2)))
       (pl3 ((act inc-action) (act parameter-action parameter-style) 
             (act parameter-action specializer-style) (jmp -> pl1 2)))
       (pl4 ((act parameter-action keyword-package-style) (jmp -> pl1 2)))
       (pl5 ((pop)))))

     (doc
      ((dc1 ((act start-action) (jmp? doc-p -> dc2 0))) ; fail
       (dc2 ((act doc-action) (pop)))))

     (slot-list
      ((sl1 ((act start-action) (act inc-action) (jmp? end-p -> sl2 0)
             (jmp-fail parse-subform-action -> sl1 2) (jmp? end-p -> sl2 0)
             (act subform-action) (act slot-name-action) (push generic-rest -> sl1 2)))
       (sl2 ((pop)))))

     (options
      ((op1 ((act start-action) (jmp? end-p -> op5 0) (jmp-fail parse-subform-action -> op1 1)
             (jmp? end-p -> op5 0) (jmp-fail get-char-action -> op1 3) 
             (jmp? options-keyword-p -> op3 0)))
       (op2 ((jmp? end-p -> op5 0) (jmp-fail parse-subform-action -> op2 0) (jmp -> op3 0)))
       (op3 ((act subform-action) (act options-keyword-action)
             (jmp? options-doc-p -> op4 0) (push generic-rest -> op2 0)))
       (op4 ((jmp-fail options-doc-action -> op4 0) (jmp -> op2 0)))
       (op5 ((pop t)))))

     (body
      ((bd1 ((act start-action) (push region :parent -> bd2 0)))
       (bd2 ((pop t)))))

     (variable-definitions
      ((vd1 ((act start-action) (act inc-action) (act next-sexp-action) (jmp? end-p -> vd3 0) 
             (jmp? variable-definitions-list-p -> vd2 0)
             (act symbol-action variable-definition-symbol-style) (jmp -> vd1 2)))
       (vd2 ((act variable-definitions-symbol-action)
             (push generic-rest) (jmp -> vd1 2)))
       (vd3 ((pop)))))

     (variable-form
      ((vf1 ((act start-action) (act variable-definitions-symbol-action)
             (push generic-rest -> vf2 0)))
       (vf2 ((pop)))))

     (struct-options
      ((so1 ((act start-action) (jmp? end-p -> so5 0) (jmp-fail parse-subform-action -> so1 1)
             (jmp? end-p -> so5 0) (jmp-fail get-char-action -> so1 3) 
             (jmp? options-keyword-p -> so3 0)))
       (so2 ((jmp? end-p -> so5 0) (jmp-fail parse-subform-action -> so2 0) (jmp -> so3 0)))
       (so3 ((act subform-action) (act options-keyword-action)
             (jmp? struct-include-p -> so4 0) (push generic-rest -> so2 0)))
       (so4 ((jmp-fail struct-include-action -> so4 0) (jmp -> so2 0)))
       (so5 ((pop t)))))

     (struct-sym
      ((ss1 ((act start-action) (act optional-paren-action) 
             (act symbol-action defstruct-symbol-style) (jmp? end-p -> ss1 5)
             (push struct-options :parent) (pop)))))

     (struct-fields
      ((sf1 ((act start-action) (jmp? end-p -> sf3 0) 
             (jmp-fail next-sexp-action -> sf1 1) (jmp? end-p -> sf3 0) 
             (jmp? struct-field-list-p -> sf2 0) (act struct-field-action) (jmp -> sf1 1)))
       (sf2 ((act struct-fields-list-action) (push generic-rest -> sf1 1)))
       (sf3 ((pop t)))))

     (embedded-function-definitions
      ((ef1 ((act start-action) (act inc-action) (jmp? end-p -> ef5 0) 
             (jmp-fail parse-subform-action -> ef1 2) (jmp? end-p -> ef5 0) (jmp -> ef3 0)))
       (ef2 ((jmp? end-p -> ef5 0) (jmp-fail parse-subform-action -> ef2 0) (jmp -> ef3 0)))
       (ef3 ((act subform-action) (act superparen-action)
             (act symbol-action embedded-function-symbol-style) (push parameter-list -> ef4 0)))
       (ef4 ((push doc) (push body :subform) (act superparen-action) (jmp -> ef2 0)))
       (ef5 ((pop)))))

     (case-match-forms
      ((cm1 ((act start-action) (jmp? end-p -> cm3 0) (jmp-fail parse-subform-action -> cm1 1)
             (jmp? end-p -> cm3 0) (act inc-action) (jmp? list-p -> cm2 0) 
             (act case-match-action case-match-symbol-style) (push body :subform) (act paren-action) 
             (jmp -> cm1 1)))
       (cm2 ((push case-match-list) (push body :subform) (act paren-action) (jmp -> cm1 1)))
       (cm3 ((pop)))))

     (case-match-list
      ((cl1 ((act start-action) (jmp? end-p -> cl2 0) (act next-sexp-action)
             (act case-match-action case-match-symbol-style) (jmp -> cl1 1)))
       (cl2 ((pop)))))

     (keyword-pairs
      ((kp1 ((act start-action) (jmp? end-p -> kp5 0) 
             (jmp-fail parse-action -> kp1 1)
             (jmp? env-lparen-p -> kp2 0) (jmp? env-lambda-p -> kp2 0) 
             (jmp? env-keyword-p -> kp3 0) (jmp? env-quote-p -> kp4 0)))
       (kp2 ((act char-action) (push region -> kp1 1)))
       (kp3 ((act char-action) (act keyword-action) (jmp -> kp1 1)))
       (kp4 ((act char-action) (act quote-action) (jmp -> kp1 1)))
       (kp5 ((pop t)))))

     (form
      ;; This must handle atoms as well as lists
      ((fr1 ((act start-action) (jmp? end-p -> fr1 3) (jmp? region-p -> fr2 0)
             (jmp? colon-p -> fr3 0) (act complete-action) (jmp -> fr4 0)))
       (fr2 ((push region) (jmp -> fr4 0)))
       (fr3 ((act keyword-action) (jmp -> fr4 0)))
       (fr4 ((pop)))))

     (loop-body
      ((lb1 ((act start-action) (jmp? end-p -> lb8 0)
             (jmp-fail parse-loop-action -> lb1 1) (jmp-fail subform-action -> lb1 2) 
             (act unstyle-action) (jmp? loop-atom-p -> lb6 0)
             (jmp? defstyle-form-p -> lb2 0) (jmp? package-form-p -> lb3 0)
             (jmp? generic-form-p -> lb4 0) (jmp? empty-loop-form-p -> lb5 0) (jmp-stop -> lb1 3)))
       (lb2 ((push defstyle-form -> lb1 1) (jmp -> lb1 3)))
       (lb3 ((push package-form -> lb1 1) (jmp -> lb1 3)))
       (lb4 ((push generic-form -> lb1 1) (jmp -> lb1 3)))
       (lb5 ((act empty-form-action) (jmp -> lb1 1)))
       (lb6 ((jmp? loop-keyword-p -> lb7 0) (jmp-fail loop-atom-action -> lb1 3) 
             (jmp -> lb1 1)))
       (lb7 ((jmp-fail loop-keyword-action -> lb1 3) (jmp -> lb1 1)))
       (lb8 ((pop)))))))

;;; ----------------------------------------------------------------------------
;;; utils and macros
;;;
#-cc-debug
(declaim (inline all-whitespace-p char-printable-p get-next-top-level-lparen-position stack-empty-p))

(defun all-whitespace-p (start end)
   (do* ((pos start (1+ pos))
          (char (buffer-char *buffer* pos) (buffer-char *buffer* pos)))
         ((>= pos end) t)
      (unless (whitespacep char) (return-from all-whitespace-p nil))))

(defun get-next-top-level-lparen-position ()
   "Find the position of the start of the next top-level form."
   (do* ((lines-in-buffer (lines-in-buffer *buffer*))
          (current-line (buffer-line *buffer*))
          (position (buffer-position *buffer*))
          (count 1 (1+ count))
          (line-start (buffer-line-start *buffer* position count)
                      (buffer-line-start *buffer* position count))
          (char (buffer-char *buffer* line-start) (buffer-char *buffer* line-start)))
         ((>= count (- lines-in-buffer current-line)))
      (when (char= char #\() (return line-start))))

;; *** this needs work
(defun char-printable-p (char)
   "Is the char printable? This includes various types of deletes, newlines, etc."
   (let ((code (char-code char)))
     ;; (format t "~%code: ~s" code)
     (let ((control-key-p (control-key-p))
           (option-key-p (option-key-p))
           (command-key-p (command-key-p)))
       (cond ((not (or control-key-p option-key-p command-key-p))
                (when (or (and (>= code 32) (<= code 127)) ; this is the primary case
                            (= code 13) ; #\Newline
                            (= code 8)  ; #\Delete, #\Backspace
                            (= code 10) ; $\Linefeed
                            (= code 127)) ; #\DEL
                   t))
               #+elvis
               ((and control-key-p option-key-p) 
                (when (or (= code 8)) ; control-meta-h & control-meta-delete ****
                   t))
               (control-key-p
                (cond ((or (= code 4) ; control-d
                             (= code 11) ; control-k
                             (= code 23)) ; control-w
                         t)
                        #+cc-debug((= code 17) ; control-q
                                   (query-stack) nil)
                        #+cc-debug((= code 26)
                                   (toggle-debug) nil))) ; control-z
               (option-key-p
                (when (or (= code 182) ; meta-d
                            (= code 202) ; meta-space ??
                            (= code 199)) ; meta-\ ??
                   t))
               (t nil)))))

(defun stack-empty-p ()
   (<= (p-stack-idx *parser*) 0))

(defMacro pop-stack ()
   `(unless (< stack-idx 3)
       (setf node (svref stack stack-idx)) (decf stack-idx)
       (setf pos (svref stack stack-idx)) (decf stack-idx)
       (setf arc-idx (svref stack stack-idx)) (decf stack-idx)
       (setf env (svref stack stack-idx)) (decf stack-idx) t))

(defMacro push-stack ()
   `(progn (incf stack-idx) (setf (svref stack stack-idx) env) 
             (incf stack-idx) (setf (svref stack stack-idx) arc-idx)
             (incf stack-idx) (setf (svref stack stack-idx) pos) 
             (incf stack-idx) (setf (svref stack stack-idx) node)))

;;; This is also set after editing with the prefs dialog.
(multiple-value-setq (*ff*  *ms*) (font-codes (getf *generic-text-style* :font)))

;;; Accessors for the grammar-description lists.
(defun network-list-name (network-list) (first network-list))
(defun network-list-nodes (network-list) (second network-list))
(defun node-list-name (node-list) (first node-list))
(defun node-list-arcs (node-list) (second node-list))
(defun arc-list-type (arc-list) (first arc-list))
(defun arc-list-action (arc-list) (second arc-list))
(defun arc-list-param (arc-list) (let ((length (length arc-list)))
                                    (when (or (= length 3) (= length 6))
                                            (third arc-list))))
(defun arc-list-idx (arc-list)  (first (last arc-list)))
(defun arc-list-prereq (arc-list) (second arc-list))
(defun arc-list-end-pos-dec-p (arc-list) (second arc-list))
(defun arc-list-network (arc-list) (second arc-list))
(defun arc-list-dest (arc-list) (when (> (length arc-list) 3) (first (last arc-list 2))))

;;; These are the arc types:
;;; (ACT ACTION)
;;; (ACT ACTION ACCESSOR)
;;; (JMP -> NODE IDX)
;;; (JMP? TEST -> NODE IDX)
;;; (JMP-STOP -> NODE IDX)
;;; (JMP-FAIL ACTION -> NODE IDX)
;;; (PUSH NETWORK -> NODE IDX)
;;; (PUSH NETWORK :END-TYPE -> NODE IDX)
;;; (PUSH NETWORK :END-TYPE)
;;; (PUSH NETWORK)
;;; (POP-FAIL ACTION)
;;; (POP-FAIL ACTION PARAM)
;;; (POP-FAIL)
;;; (POP t)
;;; (POP)

;;; ----------------------------------------------------------------------------
;;;
(defClass ARC ()
   ((node :initform nil :initarg :node :accessor a-node)
    ;; the textual rep, used by print-object
    (rep :initform nil :initarg :rep :accessor a-rep))
   (:documentation "Base class for all derived arcs."))

(defMethod initialize-instance :after ((a arc) &key arc-list)
   (setf (a-rep a) arc-list))

;;; This replaces the symbols representing networks and nodes with
;;; the actual objects, stored in the respective hash-tables.
(defMethod link-objects ((a arc) network-table node-table)
   ;(declare (ignore network-table))
   (when (a-dest-node a)
      (let ((node-object (gethash (a-dest-node a) node-table)))
        (if node-object
          (setf (a-dest-node a) node-object)
          (error "Unknown node: ~a." (a-dest-node a))))))

(defMethod print-object ((a arc) stream)
   (format stream "<Arc(~A):~A>" (n-name (a-node a)) (a-rep a)))

;;; ----------------------------------------------------------------------------
;;;
;;; (ACT ACTION)
;;; (ACT ACTION PARAM)
;;;
;;; An ACT-ARC funcalls an ACTION, supplying a PARAM if it has one.
;;;
;;; Depending on the values returned by ACTION, the arc can be repeated
;;; with the next keystroke, or repeated with the next keystroke using
;;; an inc-ed POS.  Otherwise, execution proceeds with the next arc in 
;;; the current node.
;;;
(defClass ACT-ARC (arc)
   ((action :initform nil :initarg :action :accessor a-action)
    (param :initform nil :initarg :param :accessor a-param)))

(defMethod initialize-instance :after ((a act-arc) &key arc-list)
   (setf (a-action a) (arc-list-action arc-list))
   (setf (a-param a) (arc-list-param arc-list)))

(defMethod link-objects ((a act-arc) network-table node-table)
   (declare (ignore network-table node-table))
   ()) ; do nothing

(defMethod process-arc ((a act-arc) parser)
   (with-slots (node pos arc-idx env) parser  
      (multiple-value-bind (p e)
                             (if (a-param a)
                               (funcall (a-action a) pos env (a-param a))
                               (funcall (a-action a) pos env)) 
         ;; (null p) is a flag for an unfinished action -
         ;; keep the same stack, pos, env, node and arc-idx, so
         ;; that the computation can continue with the next keystroke.

         ;; (eq p :inc-pos) is also an unfinished action -
         ;; keep the same stack, env, node and arc-idx, but increment pos
         (cond ((null p)
                 ;; restart at pos
                 (return-from process-arc nil))
                ((eq p :inc-pos) ; used by superparen-action and default-value-start-action
                 (when (< (1+ pos) (buffer-size *buffer*))
                    (incf pos))
                 (when (and *dynamic-p* (= pos *dynamic-pos*))
                    (return-from process-arc nil)))
                (t
                 (setf pos p env e)
                 (incf arc-idx)))))
   t)

;;; ----------------------------------------------------------------------------
;;;
;;; (JMP -> DEST-NODE DEST-IDX)
;;;
;;; JMP-ARC is an unconditional jump to DEST-NODE at DEST-IDX.
;;; Only jump to another node within the same network.
;;;
(defClass JMP-ARC (arc)
   ((dest-index :initform nil :initarg :index :accessor a-dest-index)
    (dest-node :initform nil :initarg :destination :accessor a-dest-node)))

(defMethod initialize-instance :after ((a jmp-arc) &key arc-list)
   (setf (a-dest-index a) (arc-list-idx arc-list))
   (setf (a-dest-node a) (arc-list-dest arc-list)))

(defMethod process-arc ((a jmp-arc) parser)
   (with-slots (node arc-idx) parser 
      (setf node (a-dest-node a))
      (setf arc-idx (a-dest-index a)))
   t)

;;; ----------------------------------------------------------------------------
;;;
;;; (JMP? TEST ->  DEST-NODE DEST-IDX)
;;;
;;; JMP?-ARC is a conditional jump to DEST-NODE at DEST-IDX.  If TEST
;;; returns true, the jump is executed.  Otherwise, execution proceeds
;;; with the next arc in the current node.  TEST should be a simple 
;;; predicate, since POS and ENV will not be affected.
;;;
(defClass JMP?-ARC (jmp-arc)
   ((prereq :initform nil :initarg :prereq :accessor a-prereq)))

(defMethod initialize-instance :after ((a jmp?-arc) &key arc-list)
   (setf (a-prereq a) (arc-list-prereq arc-list)))

(defMethod process-arc ((a jmp?-arc) parser)
   (with-slots (node arc-idx pos env) parser 
      (if (funcall (a-prereq a) pos env)
        (setf node (a-dest-node a) arc-idx (a-dest-index a))
        (incf arc-idx)))
   t)

;;; ----------------------------------------------------------------------------
;;;
;;; (JMP-FAIL ACTION ->  DEST-NODE DEST-IDX)
;;; (JMP-FAIL ACTION PARAM ->  DEST-NODE DEST-IDX)
;;;
;;; JMP-FAIL-ARC is a conditional jump to DEST-NODE at DEST-IDX.  
;;; If ACTION fails, the jump is executed.  otherwise, execution proceeds
;;; with the next arc in the current node.  
;;; 
;;; Note that this arc takes an ACTION, unlike JMP? which takes a predicate.
;;;
(defClass JMP-FAIL-ARC (jmp-arc)
   ((param :initform nil :initarg :param :accessor a-param)
    (action :initform nil :initarg :action :accessor a-action)))

(defMethod initialize-instance :after ((a jmp-fail-arc) &key arc-list)
   (setf (a-param a) (arc-list-param arc-list))
   (setf (a-action a) (arc-list-action arc-list)))

(defMethod process-arc ((a jmp-fail-arc) parser)
   (with-slots (node arc-idx pos env) parser 
      (multiple-value-bind (p e)
                             (if (a-param a)
                               (funcall (a-action a) pos env (a-param a))
                               (funcall (a-action a) pos env))
         (cond ((null e)
                 ;; (null e) is a flag for failure
                 (setf node (a-dest-node a))
                 (setf arc-idx (a-dest-index a))
                 ;; (not (null p)) means to use p for position,
                 (when p (setf pos p))
                 (when *dynamic-p* 
                    (unless p ; p is set when p < *dynamic-pos* -- keep processing
                       (return-from process-arc nil))))
                (t ; no failure, just set the new values for POS and ENV and continue
                 ;; with the next arc.
                 (setf pos p env e) 
                 (incf arc-idx)))))
   t)

;;; ----------------------------------------------------------------------------
;;;
;;; (JMP-STOP ->  DEST-NODE DEST-IDX)
;;;
;;; JMP-STOP-ARC is an unconditional jump to DEST-NODE at DEST-IDX.
;;; When styling incrementally, this forces the next keystroke by returning nil.
;;;
(defClass JMP-STOP-ARC (jmp-arc) ())

(defMethod process-arc ((a jmp-stop-arc) parser)
   (with-slots (node arc-idx) parser 
      ;(declare (fixnum pos arc-idx) (optimize (speed 3)(safety 0)))
      (setf node (a-dest-node a))
      (setf arc-idx (a-dest-index a))
      (when *dynamic-p*
        (return-from process-arc nil))
      t))

;;; ----------------------------------------------------------------------------
;;;
;;; (PUSH NETWORK -> DEST-NODE DEST-IDX)
;;; (PUSH NETWORK :END-TYPE -> DEST-NODE DEST-IDX)
;;; (PUSH NETWORK :END-TYPE)
;;; (PUSH NETWORK)
;;;
;;; A PUSH-ARC pushes the current NODE, ARC-IDX, POS, and ENV onto the stack.
;;; The start node of NETWORKbecomes the new node. arc-idx is set to 0, and a new
;;; env is created.  
;;;
;;; If :END-TYPE is specified, it is used to calculate the new :END value for the 
;;; pushed NETWORK.
;;;
;;; When the PUSH-ARC is POPPED, the network continues with DEST-NODE and DEST-IDX.
;;; If no DEST-IDX and DEST-NODE are specified, the parse continues, after the POP, with the next arc 
;;; after the PUSH-ARC, but uses the new inc-ed value of pos.
;;;
(defClass PUSH-ARC (arc)
   ((network :initform nil :initarg :network :accessor a-network)
    ;; param is the :end-type value (:form :parent :subform) used by 
    ;; start-action to create the appropriate :end value.
    (param :initform nil :initarg :param :accessor a-param)
    (dest-index :initform nil :initarg :index :accessor a-dest-index)
    (dest-node :initform nil :initarg :node :accessor a-dest-node)))

(defMethod initialize-instance :after ((a push-arc) &key arc-list)
   (setf (a-network a) (arc-list-network arc-list))
   (setf (a-param a) (arc-list-param arc-list))
   (setf (a-dest-index a) (arc-list-idx arc-list))
   (setf (a-dest-node a) (arc-list-dest arc-list)))

(defMethod link-objects :after ((a push-arc) network-table node-table)
   (declare (ignore node-table))
   (unless (eql (a-network a) 'defstyle-network) ; if defstyle-network, don't modified
      (let ((network-object (gethash (string (a-network a)) network-table)))
        (if network-object
          (setf (a-network a) network-object)
          (error "Unknown network: ~a." (a-network a))))))

(defMethod process-arc ((a push-arc) parser)
   (with-slots (node arc-idx pos env stack stack-idx) parser 
      (let* ((start (get-env-value env :subform-start))
             (defstyle-network (get-env-value env :defstyle-network))
             (new-env (list (list :parent-end (get-env-value env :end)))))
        (push-stack)
        (when start
           (setf new-env (set-env-value new-env :subform-end (build-end start))))
        (setf env new-env arc-idx 0)
        (when (a-param a)
           ;; :end-type is used in start-action - if no param, :form is the default.
           ;; :end-type determines how the subform :end value is calculated.
           (setf env (set-env-value env :end-type (a-param a))))
        (if (typep (a-network a) 'network)
          (setf node (start-node (a-network a)))
          (setf node (start-node defstyle-network)))))
   t)

;;; ----------------------------------------------------------------------------
;;; (POP)
;;; (POP t)
;;;
;;; A POP-ARC pops the stack.
;;; If end-pos-dec-p is t, position is set to (1- (get-end env)), otherwise (get-end env).
;;;
(defClass POP-ARC (arc)
   ((end-pos-dec-p :initform nil :initarg :end-pos-dec-p :accessor a-end-pos-dec-p)))

(defMethod initialize-instance :after ((a pop-arc) &key arc-list)
   (setf (a-end-pos-dec-p a) (arc-list-end-pos-dec-p arc-list)))

(defMethod link-objects ((a pop-arc) network-table node-table)
   (declare (ignore network-table node-table))
   ()) ; do nothing

(defMethod process-arc ((a pop-arc) parser)
   (with-slots (node arc-idx pos env stack stack-idx) parser 
      (cond ((a-end-pos-dec-p a)
              (setf pos (1- (get-end env))))
             (t
              (setf pos (get-end env))))
      (let ((new-pos pos)) ; save the inc-ed pos
        (if (null (pop-stack)) 
          (return-from process-arc nil)
          (let ((arc (aref (n-arcs node) arc-idx)))
            (cond ((and (a-dest-node arc) (a-dest-index arc))
                    (setf node (a-dest-node arc)
                          pos new-pos arc-idx (a-dest-index arc)))
                   (t
                    ;; a push without a destination, just continue, but use new pos
                    (setf pos new-pos)
                    (incf arc-idx)))))))
   t)

;;; ----------------------------------------------------------------------------
;;;
;;; (POP-FAIL ACTION)
;;; (POP-FAIL ACTION PARAM)
;;; (POP-FAIL)
;;;
;;; This is a conditional POP.
;;; POP-FAIL-ARC funcalls ACTION, supplying a PARAM if it has one.
;;; Depending on the values returned by the ACTION, the POP-FAIL-ARC
;;; may pop the stack.  If the stack is popped, this represents
;;; a failure of the previous PUSH and the destination of the previous
;;; PUSH is ignored (if it has a destination).  The arc-idx of the PUSH 
;;; node is inc-ed and processing proceeds with the next arc.
;;;
;;; If ACTION does not fail, new values of POS and ENV are set
;;; and the next arc in the current node is executed.
;;;
;;; If the POP-FAIL-ARC has no ACTION, pop the stack and inc the 
;;; calling node's arc-idx, ignoring any destination node.  This 
;;; is the how a failure is propagated back up the stack.
;;;
(defClass POP-FAIL-ARC (arc)
   ((action :initform nil :initarg :action :accessor a-action)
    (param :initform nil :initarg :param :accessor a-param)))

(defMethod initialize-instance :after ((a pop-fail-arc) &key arc-list)
   (setf (a-action a) (arc-list-action arc-list))
   (setf (a-param a) (arc-list-param arc-list)))

(defMethod link-objects ((a pop-fail-arc) network-table node-table)
   (declare (ignore network-table node-table))
   ()) ; do nothing

(defMethod process-arc ((a pop-fail-arc) parser)
   (with-slots (node arc-idx pos env stack stack-idx) parser 
      (cond ((a-action a)
              (multiple-value-bind (p e)
                                     (if (a-param a)
                                       (funcall (a-action a) pos env (a-param a))
                                       (funcall (a-action a) pos env))
                 (cond ((null e)
                         ;; (null e) is a flag for failure.
                         ;; Pop the stack and inc the arc-idx
                         ;; of the calling network
                         (pop-stack)
                         (incf arc-idx)
                         ;; (not (null p)) means to use p for position,
                         (when p (setf pos p))
                         (when *dynamic-p* 
                            (unless p ; p is set when p < *dynamic-pos* -- keep processing
                               (return-from process-arc nil))))
                        (t ; Success - Just use the new values for POS and ENV and continue
                         ;; with the next arc.
                         (setf pos p env e) 
                         (incf arc-idx))))
              t)
             (t
              (pop-stack)
              (incf arc-idx)))))

(defConstant %max-arcs% 15)

;;; ----------------------------------------------------------------------------
;;;
(defClass NODE ()
   ((name :initform nil :initarg :name :accessor n-name)
    (network :initform nil :initarg :network :accessor n-network)
    ;; an array of arc objects
    (arcs :initform  nil :accessor n-arcs)
    (num-arcs :initform nil :accessor n-num-arcs)))

(defMethod initialize-instance :after ((n node) &key node-description)
   (setf (n-arcs n) (make-array %max-arcs%)) 
   (setf (n-name n) (node-list-name node-description))
   (let ((index 0)
         type)
     (dolist (arc-list (node-list-arcs node-description))
        (setf type (find-symbol (string-upcase (concatenate 'string (string (arc-list-type arc-list)) "-ARC")) :cc))
        (assert type)
        (setf (aref (n-arcs n) index) (make-instance type :arc-list arc-list :node n))
        (incf index))
     (setf (n-num-arcs n) index)))

(defMethod link-objects ((n node) network-table node-table)
   (dotimes (index (n-num-arcs n))
      (link-objects (aref (n-arcs n) index) network-table node-table)))

(defMethod print-object ((n node) stream)
   (format stream "<Node:~A>" (n-name n)))

;;; *parser*

;;; ----------------------------------------------------------------------------
;;;
(defClass NETWORK ( )
   ((name :initform nil :initarg :name :accessor nw-name)
    ;; the textual description is saved here for debugging
    (name-symbol :initform nil :accessor nw-name-symbol)
    (network-description :initform nil :initarg :network-description :accessor nw-network-description)
    ;; a list of node objects; the start-node is in position 0
    (nodes :initform nil :initarg :nodes :accessor nw-nodes)))

(defMethod initialize-instance :after ((nw network) 
                                       &key network-description node-table)
   (setf (nw-name nw) (string (network-list-name network-description)))
   (setf (nw-name-symbol nw) (find-symbol (nw-name nw) :cc))
   (let (new-node nodes)
     (dolist (node-descp (network-list-nodes network-description))
        (setf new-node (make-instance 'node 
                          :network nw 
                          :node-description node-descp))
        ;; node name must be unique
        (when (gethash (n-name new-node) node-table)
           (error "Duplicate node name: ~a" (n-name new-node)))
        (setf (gethash (n-name new-node) node-table) new-node)
        (setf nodes (nconc nodes (list new-node))))
     (setf (nw-nodes nw) nodes)))

(defMethod link-objects ((nw network) network-table node-table)
   (dolist (node (nw-nodes nw))
      (link-objects node network-table node-table)))

(defMethod start-node ((nw network))
   (first (nw-nodes nw)))

(defMethod print-object ((nw network) stream)
   (format stream "<Net:~(~s~)>" (nw-name nw)))

;;; ----------------------------------------------------------------------------
;;;
(defClass GRAMMAR ()
   ((name :initform nil :initarg :name :accessor g-name)
    ;; a hash-table of all the nodes in the grammar
    (nodes :initform nil :initarg :nodes :accessor g-nodes)
    ;; a hash-table of the defstyle networks in the grammar
    (defstyle-networks :initform nil :accessor g-defstyle-networks)
    ;; a hash-table of the non-defstyle networks in the grammar
    (networks :initform nil :accessor g-networks)
    ;; a hash-table of loop-keyword
    (loop-keywords :initform nil :accessor g-loop-keywords)))

;;; Compile the textual description of the grammar into Lisp objects.
(defMethod initialize-instance :after ((g grammar)  &key grammar-description)
   (let (networks)
     (setf (g-nodes g) (make-hash-table))
     (setf (g-defstyle-networks g) (make-hash-table :test 'equal))
     (setf (g-networks g) (make-hash-table :test 'equal))
     (setf (g-loop-keywords g) (make-hash-table :test 'equal))
     (dolist (network-descp grammar-description)
        (push (add-network g network-descp) networks))
     ;; substitute the actual node and network objects for the references in the arcs
     (dolist (network networks)
        (link-objects network (g-networks g) (g-nodes g)))
     (dolist (keyword '(above across always and append appending by collect collecting count 
                        counting do doing downfrom downto each else end external-symbol 
                        external-symbols finally for from hash-key hash-keys hash-value 
                        hash-values if in into initially loop-finish maximize maximizing 
                        minimize minimizing named nconc nconcing never of on present-symbol 
                        present-symbols repeat return sum summing symbol symbols the then 
                        thereis to unless until upfrom upto using when while with))
        (setf keyword (string-upcase (string keyword)))
        (when (gethash keyword (g-loop-keywords g))
           (error "Duplicate keyword: ~a" keyword))
        (setf (gethash  keyword (g-loop-keywords g)) keyword))))

(defMethod add-network ((g grammar) network-descp &optional (table (g-networks g)))
   (let ((network (make-instance 'network 
                     :node-table (g-nodes g)
                     :network-description network-descp)))
     (when (gethash (nw-name network) table)
        (error "Duplicate network name: ~a" (nw-name network)))
     (setf (gethash (nw-name network) table) network)))

(defMethod add-network-and-link ((g grammar) network-descp)
   (link-objects (add-network g network-descp (g-defstyle-networks g)) (g-networks g) (g-nodes g)))

(defMethod get-network ((g grammar) nw-name)
   (gethash (string-upcase nw-name) (g-networks g)))

(defMethod get-defstyle-network ((g grammar) nw-name)
   (gethash (string-upcase nw-name) (g-defstyle-networks g)))

(defMethod loop-keywd-p ((g grammar) name)
   (gethash (string-upcase name) (g-loop-keywords g)))

(defMethod print-object ((g grammar) stream)
   (format stream "<G:~(~s~)>" (g-name g)))

(defParameter *rtn-grammar* (make-instance 'grammar 
                                 :name "RTN Grammar"
                                 :grammar-description *rtn-grammar-description*))

;;; ----------------------------------------------------------------------------
;;;
(defClass PARSER ()
   ((node :initform nil :accessor p-node) ; the node currently being processed
    (pos :initform nil :accessor p-pos) ; parsing position in *buffer*, usually not the same as *dynamic-pos*
    (arc-idx :initform 0 :accessor p-arc-idx) ; the index into current-node's arc array
    (env :initform nil :accessor p-env) ; an a-list of values used by the current network 

    ;; This is a stack of the four slot values listed above. At any
    ;; point the stack specifies the pending computation.
    ;; When styling incrementally, it is saved and the computation
    ;; can continue where it left off, without re-parsing the 
    ;; entire top-level form. 
    (stack :initform nil :accessor p-stack)
    (stack-idx :initform -1 :accessor p-stack-idx)

    ;; The value of the previous *dynamic-pos*.
    ;; It is used to determine if the parse has a continuation.
    (previous-dynamic-pos :initform nil :accessor p-previous-dynamic-pos)
    ;; The position of the start of the top-level form being parsed.
    (top-level-start-pos :initform nil :accessor p-top-level-start-pos)
    ;; The scroller-view of the file being styled.  If this changes, initialize.
    (scroller-view :initform nil :accessor p-scroller-view)
    ;; flags for special cases
    (odd-quotes-p :initform nil :accessor p-odd-quotes-p)
    (new-context-p :initform nil :accessor p-new-context-p) ; ***
    (inside-semi-colon-comment-p :initform nil :accessor p-inside-semi-colon-comment-p)))

(defMethod initialize-instance :after ((p parser) &key)
   (setf (p-stack p) (make-array 400)))

(defMethod print-object ((p parser) stream)
   (format stream "<Parser>"))

#-cc-debug
(declaim (inline calculate-context continuation-p odd-quotes-p shrink-stack stack-backup-p 
                  init-parse get-top-level-start-pos))

(defun init-parse (p)
   ; (setf *segment-list* nil)
   ; (setf *segment-array* nil)
   (setf (p-node p) nil)
   ; (setf (p-pos p) nil)
   (setf (p-arc-idx p) 0)
   (setf (p-stack-idx p) -1)
   (setf (p-previous-dynamic-pos p) nil)
   (setf (p-top-level-start-pos p) nil)
   (setf (p-odd-quotes-p p) nil)
   (setf (p-inside-semi-colon-comment-p p) nil))

(defun get-top-level-start-pos (p)
   (calculate-context p)
   (when (p-top-level-start-pos p)
      (let* ((end-sharp-comment (ccl::buffer-forward-search *buffer* "|#" (p-top-level-start-pos p)))
             (begin-sharp-comment (when end-sharp-comment
                                     (ccl::buffer-backward-search *buffer* "#|" end-sharp-comment))))
        ;; embedded?
        (cond ((and begin-sharp-comment end-sharp-comment
                      (< begin-sharp-comment (p-top-level-start-pos p) end-sharp-comment))
                 ;; (format t "~%embedded")
                 (init-parse p)  nil)
                (t
                 (p-top-level-start-pos p))))))

;;; I hate strings and comments -
(defun calculate-context (p  &optional new-char)
   "Calculate p-top-level-start-pos and set various special case flags."
   (setf (p-new-context-p p) nil)
   (setf *segment-list* nil)
   (setf *segment-array* nil)
   (setf (p-inside-semi-colon-comment-p p) (when new-char (char= new-char #\;)))
   (do* ((pos (max (1- (buffer-position *buffer*)) 0))
          (char (buffer-char *buffer* pos) (buffer-char *buffer* pos))
          (char-1 (buffer-char *buffer* (max (1- pos) 0))
                  (buffer-char *buffer* (max (1- pos) 0)))
          (count (if new-char (if (char= new-char #\") 1 0) 0))
          (first-char-p t nil)
          line-start-p semi-colon-pos right-quote-pos left-quote-pos)
         ((and (char= char #\() (or (char-eolp char-1) (= pos 0)))
          (setf (p-top-level-start-pos p) pos)
          (cond ((= (mod count 2) 0) 
                   (when (and right-quote-pos left-quote-pos semi-colon-pos)
                      (if (< left-quote-pos semi-colon-pos right-quote-pos)
                        (setf (p-inside-semi-colon-comment-p p) nil)
                        (setf (p-inside-semi-colon-comment-p p) t)))
                   (setf (p-odd-quotes-p p) nil))
                  (t 
                   (if (and semi-colon-pos left-quote-pos)
                     (cond ((< left-quote-pos semi-colon-pos)
                              (setf (p-inside-semi-colon-comment-p p) nil)
                              (setf (p-odd-quotes-p p) t))
                             (t
                              (setf (p-odd-quotes-p p) nil)))
                     (setf (p-odd-quotes-p p) t)))))
      (cond ((and (char-eolp char) (not first-char-p))
               (setf line-start-p t))
              ((and (char= char #\;) (not line-start-p) (not (char= char-1 #\\)))
               (setf (p-inside-semi-colon-comment-p p) t)
               (setf semi-colon-pos pos))
              ((and (char= char #\") (not (char= char-1 #\\)))
               (incf count)
               (unless right-quote-pos (setf right-quote-pos pos))
               (setf left-quote-pos pos)))
      (decf pos)
      (when (< pos 0)
         (setf (p-top-level-start-pos p) nil)
         (if (= (mod count 2) 0) 
           (return-from calculate-context (setf (p-odd-quotes-p p) nil))
           (return-from calculate-context (setf (p-odd-quotes-p p) t))))))

(defun continuation-p (p  &optional before-char-inserted-p new-char)
   "A continuation exists when chars are typed sequentially"
   ;; *** delete 
   (let ((pos (cond ((and before-char-inserted-p new-char)
                      (if (char= new-char #\delete)
                        (1- (buffer-position *buffer*))
                        (1+ (buffer-position *buffer*))))
                     (t *dynamic-pos*))))
     (and (p-previous-dynamic-pos p) (p-node p) 
           (or (= pos (1+ (p-previous-dynamic-pos p)))
                (and (> pos (p-previous-dynamic-pos p))
                      (all-whitespace-p (p-previous-dynamic-pos p) (1- pos)))))))

(defun odd-quotes-p (p new-char)
   (cond ((or (p-new-context-p p) (char= new-char #\"))
            (calculate-context p new-char))
           ((not (continuation-p p t new-char))
            (calculate-context p new-char)))
   #-cc-debug(p-odd-quotes-p p)
   #+cc-debug(values (p-odd-quotes-p p) (p-inside-semi-colon-comment-p p)))

;;; When doing an arbitrary edit on a string or comment, all the stack frames with a 
;;; start value greater than *dynamic-pos* are potentially hosed, since the
;;; edit may shrink or expand the form, invalidating their start position.
;;; Decrement stack-idx as required, so that subsequent stack-backups will be valid.
;;; The segment array is also potentially hosed - initialize it.
(defun shrink-stack (p)
   (setf *segment-list* nil)
   (setf *segment-array* nil)
   (with-slots (stack stack-idx) p
      (loop
        (when (< stack-idx 3) (return))
        (let* ((stack-env (svref stack (- stack-idx 3)))
               (stack-start (get-env-value stack-env :start)))
          (when (and stack-start (< stack-start *dynamic-pos*)) (return))
          (decf stack-idx 4)))))

;;; When moving to edit an arbitrary position within a form, check to see if
;;; there is a stack frame with (<= start *dynamic-pos* end).  If there is,
;;; we can restart there, and avoid parsing the entire form - cool.
(defun stack-backup-p (p)
   "Not a pure predicate.  This pops the stack, looking for a restart position."
   (with-slots (node pos arc-idx env stack stack-idx) p
      (let ((start (get-env-value env :start))
            (end (get-end env)))
        ;;        (format t "~%stack index: ~s" stack-idx)
        ;;        (format t "~%start: ~S" start)
        ;;        (format t "~%end: ~S" end)
        ;;        (format t "~%*dynamic-pos*: ~S" *dynamic-pos*)
        (when (and start end node)
           (loop
             (when (<= start *dynamic-pos* end)
                ;; If in generic-rest, backup one more frame
                (unless (eq (nw-name-symbol (n-network node)) 'generic-rest)
                   (calculate-context p (buffer-char *buffer*))
                   (setf pos start arc-idx 0 node (start-node (n-network node))
                         env (remove-if-not #'(lambda (entry)
                                                 (member (first entry)
                                                            '(:start :parent-end :end-type 
                                                              :subform-end :defstyle-network)))
                                             env))
                   (return t)))
             (when (< stack-idx 3) (return nil))
             (setf node (svref stack stack-idx)
                   env (svref stack (- stack-idx 3))
                   start (get-env-value env :start)
                   end (get-end env))
             (unless (and start end) (return nil))
             (decf stack-idx 4))))))

;;; If typing in continuously, we can restart the parse where we left off.
;;; If doing an arbitrary edit, check for a stack backup.
;;; Otherwise initialize and parse the top-level form.
(defMethod parse ((p parser) start end)
   (with-slots (node pos arc-idx env stack stack-idx prev-pos) p
      (cond ((continuation-p p)
               #+cc-debug (when *verbose-p* (format t "~%Continuation: ~S~%" (p-previous-dynamic-pos p))))
              ((stack-backup-p p)
               #+cc-debug (when *verbose-p* (format t "~%Stack Backup:~%") 
                                  (when *stack-p* (dump-stack p (aref (n-arcs node) arc-idx)))))
              (t
               #+cc-debug (when *verbose-p* (format t "~%Initializing~%"))
               (init-parse p) ; ***
               (if start
                 (setf pos start)
                 (unless (setf pos (get-top-level-start-pos p))
                    (return-from parse nil)))
               (setf node (start-node (get-network *rtn-grammar* "REGION"))
                     env (list (list :parent-end end) (list :end-type :parent)))))
      (let (arc)
        (loop
          (setf arc (aref (n-arcs node) arc-idx))
          #+cc-debug (when *debug-p* (print-info p arc))
          (unless (process-arc arc p) 
             (return-from parse *dynamic-pos*))
          (while (>= arc-idx (n-num-arcs node)) ; *** 
            (unless (pop-stack) 
               #+cc-debug (when *verbose-p* (format t "~%~%No continuation, empty stack."))
               (return-from parse nil))
            (incf arc-idx))))))

(setf *parser* (make-instance 'parser))

(defun rtn-style-buffer (&key start (end  #'(lambda () (buffer-size *buffer*))))
   (parse *parser* start end))

(defun construct-network (name description &optional (list-only-p nil))
   "Build an RTN network, based on the defstyle description list."
   (let ((node-counter 0)
         (current-arc-list (list (list 'act 'start-action)))
         current-node next-node
         nodes-list)
     (labels ((next-node ()
                 (incf node-counter)
                 (setf next-node (intern (concatenate 'string name 
                                                       (format nil "~A" node-counter)))))
              (add-act-arc (entry &optional param)
                 (let ((name (intern (concatenate 'string (string entry) "-ACTION"))))
                   (if param
                     (push (list 'act name param) current-arc-list)
                     (push (list 'act name) current-arc-list))))
              (add-pop-fail-arc (entry param)
                 (let ((name (intern (concatenate 'string (string entry) "-ACTION"))))
                   (if param
                     (push (list 'POP-FAIL name param) current-arc-list)
                     (push (list 'POP-FAIL name) current-arc-list))))
              (add-simple-push-arc (entry)
                 (push (list 'push entry) current-arc-list))
              (add-push-arc (entry)
                 (push (list 'push entry '-> (next-node) 0) current-arc-list)
                 (push (list current-node (nreverse current-arc-list)) nodes-list)
                 (setf current-arc-list nil)
                 (setf current-node next-node))
              (add-push-parent-arc (entry)
                 (push (list 'push entry :parent '-> (next-node) 0) current-arc-list)
                 (push (list current-node (nreverse current-arc-list)) nodes-list)
                 (setf current-arc-list nil)
                 (setf current-node next-node)))
       (setf current-node (next-node))
       (dolist (entry description)
          (case entry
             (ancestor (add-act-arc entry))
             (body (add-push-parent-arc entry))
             (case-match-forms (add-push-parent-arc entry))
             (derivation-list (add-act-arc entry))
             (doc (add-simple-push-arc entry))
             (embedded-function-definitions (add-push-arc entry))
             (eval-when-superparen (add-act-arc entry))
             (form (add-push-arc entry))
             (keyword-pairs (add-push-parent-arc entry))
             (loop-body (add-push-parent-arc entry))
             (loop-superparen (add-act-arc entry))
             (macro (let ((accessor (intern (concatenate 'string (string name) "-STYLE"))))
                       (add-pop-fail-arc entry accessor)))
             (optional-paren (add-act-arc entry))
             (options (add-push-parent-arc entry))
             (parameter-list (add-push-arc entry))
             (paren (add-act-arc entry))
             (qualifier (add-act-arc entry))
             (slot-list (add-push-arc entry))
             (struct-sym (add-push-arc entry))
             (struct-fields (add-push-parent-arc entry))
             (superparen (add-act-arc entry))
             (symbol (let ((accessor (intern (concatenate 'string (string name) "-SYMBOL-STYLE"))))
                        (add-act-arc entry accessor)))
             (variable-definitions (add-push-arc entry))
             (variable-form (add-push-arc entry))
             (variable-list (add-act-arc entry))
             (t (error "Bogus component in defstyle call: ~A" entry))))
       (push (list 'pop) current-arc-list)
       (push (list current-node (nreverse current-arc-list)) nodes-list)
       (if list-only-p
         (pprint (nreverse nodes-list) t)
         (add-network-and-link *rtn-grammar* (list name (nreverse nodes-list)))))))

;;; (construct-network "DEFUN" '(superparen macro symbol parameter-list doc body superparen) t)

;;; (construct-network "DOTIMES" '(paren macro variable-form body paren) t)

;;; (construct-network "DEFMETHOD" '(superparen macro symbol qualifier parameter-list doc body superparen) t)

;;; ----------------------------------------------------------------------------
;;; The interface for the incremental styling code:
;;; ----------------------------------------------------------------------------
;;;
;;; CC requires complete forms in order for the styling alogrithms to work.  
;;; As you type in a function definition, you only have a complete form when 
;;; you are finished.  This function pads the end of partially completed forms 
;;; with enough right parens to satisfy the styling algorithms.  
;;; Once the code is styled, the extra parens are deleted.
;;;
(defun dynamically-style-buffer (fred &optional style-strings-p style-semi-colon-comments-p)
   (let* ((*dynamic-p* t)
          (*buffer* (fred-buffer fred))
          (*dynamic-pos* (buffer-position *buffer*))
          (start (or (p-top-level-start-pos *parser*) 0))
          (comment-end (or (get-next-top-level-lparen-position) (buffer-size *buffer*)))
          (atom-start (atom-start (1- *dynamic-pos*)))
          (atom-end (or (atom-end *dynamic-pos*) *dynamic-pos*))
          (style-end (+ atom-end 37))
          (char (buffer-char *buffer* (max 0 (- *dynamic-pos* 1)))))
     ; (format t "~%char: ~S" char)
     ; (format t "~%atom-start: ~s" atom-start)
     ; (format t "~%atom-end: ~s" atom-end)
     (cond ((and (char= (buffer-char *buffer* (max 0 (- *dynamic-pos* 2))) #\|) 
                   (char= char #\#))
              (view-style-buffer-comments fred 0 comment-end)
              (set-previous-dynamic-pos *dynamic-pos*))
             (t
              (cond ((p-odd-quotes-p *parser*) ; inside a string
                       (shrink-stack *parser*) ; a re-edited string can cause trouble for stack-backup
                       (cond ((= (char-code char) 8) ; backspace *** delete, etc??
                                (set-previous-dynamic-pos nil))
                               (t 
                                (set-previous-dynamic-pos *dynamic-pos*))))
                      (t
                       (when (or style-strings-p style-semi-colon-comments-p)
                          (dynamically-style-buffer-comments fred start comment-end (max 0 (1- *dynamic-pos*)) 
                                                             style-strings-p style-semi-colon-comments-p))
                       (cond ((p-inside-semi-colon-comment-p *parser*)
                                (shrink-stack *parser*) ; a re-edited comment can cause trouble for stack-backup
                                (set-previous-dynamic-pos *dynamic-pos*))
                               (t
                                (unwind-protect
                                   (progn
                                      (unless (member char '(#\( #\) #\" #\space #\;))
                                         (when atom-start (style-region *generic-text-style* atom-start atom-end nil)))
                                      (buffer-insert *buffer* " o ))))))))))))))))))))))))))))))))))" atom-end)
                                      (set-previous-dynamic-pos
                                       ;; This is not a pref. It is an internal flag used to do incremental styling
                                       ;; with the RTN parser.  The default is true, since the RTN is much better
                                       ;; at incremental styling than the recursive descent algorithm, but the 
                                       ;; recursive descent algorithm still works (slowly).
                                       (if *rtn-incremental-styling-p*
                                         (rtn-style-buffer :end #'(lambda () (+ (or (atom-end *dynamic-pos*) *dynamic-pos*) 37)))
                                         (style-buffer start style-end))))
                                   (buffer-delete *buffer* atom-end style-end))))))))))

(defMethod view-activate-event-handler :after ((fred window-fred-item))
   (unless (typep fred 'ccl::listener-fred-item)
      (let* ((file-name (window-filename (view-window fred)))
             (window-title (window-title (view-window fred)))
             (type (when file-name (pathname-type file-name))))
        ;; This assumes that all new windows are going to be Lisp files, which is not
        ;; quite right.  It does turn off styling for non-lisp files which already have
        ;; a non-lisp pathname-type.  If you intend to use Fred to edit a lot on non-lisp
        ;; files, turn off dynamic styling in the prefs dialog.
        (cond ((or (and type (string= type "lisp")) (string-equal window-title "New"))
                 (setf *lisp-file-p* t)
                 (unless (eq (p-scroller-view *parser*) (view-container fred))
                    (init-parse *parser*)
                    (setf (p-scroller-view *parser*) (view-container fred))))
                (t 
                 ;; no need to style the grocery list.
                 (setf *lisp-file-p* nil))))))

(defMethod view-key-event-handler :around ((fred window-fred-item) char)
   (if (or (typep fred 'ccl::listener-fred-item) (not (char-printable-p char))
           (ccl::window-buffer-read-only-p fred) (not *do-dynamic-styling*) (not *lisp-file-p*)
           (eq (ccl::fred-shadowing-comtab fred) ccl::*i-search-comtab*)) ;i-search
     (call-next-method)
     (let* ((*buffer* (fred-buffer fred))
            (buffer-position (buffer-position *buffer*))
            (char-1 (buffer-char *buffer* (max (1- buffer-position) 0)))
            (char+1 (buffer-char *buffer* (min (buffer-size *buffer*) (1+ buffer-position))))
            (char-code (char-code char))
            restyle-comment-p style-strings-p  selection-start selection-end)
       (unless (or (odd-quotes-p *parser* char) (p-inside-semi-colon-comment-p *parser*))
          (ccl::set-buffer-insert-font-codes *buffer* *ff* *ms*))
       (cond ((and (= char-code 8) ; backspace, delete
                     (char= char-1 #\;)
                     (not (p-odd-quotes-p *parser*)))
                (setf restyle-comment-p t))
               ((and (char= (buffer-char *buffer* buffer-position) #\;)
                      (not (p-odd-quotes-p *parser*))
                      (or (= char-code 127) ; #\DEL
                           (= char-code 4))) ; control d
                (setf restyle-comment-p t)))
       (multiple-value-setq (selection-start selection-end) (selection-range (view-window fred)))
       (call-next-method) ; insert the char
       (when (> (buffer-size *buffer*) 0)
          (cond (restyle-comment-p
                   (let* ((line-start (buffer-line-start *buffer*))
                          (line-end (buffer-line-end *buffer*))
                          (start (or (ccl::ed-top-level-sexp-start-pos *buffer*) 0))
                          (end  (get-next-top-level-lparen-position)))
                     (when (and line-start line-end)
                        (style-region *generic-text-style* line-start line-end))
                     (unless end (setf end (buffer-size *buffer*)))
                     (view-style-buffer-comments fred start end)
                     (style-buffer start end)
                     (fred-update fred)))
                  (t ; *** this needs work
                   (case char-code
                      ;; *** consolidate these two:
                      ((127 8)
                       (when (or (char= char-1 #\") (char= char-1 #\;) ; delete & backspace and control-d
                                   (null (p-pos *parser*)) ; switching windows
                                   (not (= selection-start selection-end)))
                          (setf style-strings-p t)
                          (calculate-context *parser*)))
                      ;; *** #\end is also 4
                      (4 (when (or (char= char+1 #\") (char= char+1 #\;) ; #\DEL
                                     (null (p-pos *parser*)) ; switching windows
                                     (not (= selection-start selection-end)))
                            (setf style-strings-p t)
                            (calculate-context *parser*)))
                      ((13 10) (setf (p-inside-semi-colon-comment-p *parser*) nil)) ; newline and linefeed
                      (59 (when (not (p-odd-quotes-p *parser*)) ; semi-colon
                              (let* ((buffer-position (buffer-position *buffer*))
                                     (char-2 (when (and (>= buffer-position 3) (>= (buffer-size *buffer*) 3))
                                                (buffer-char *buffer* (- buffer-position 3))))
                                     (char-1 (when char-2
                                                (buffer-char *buffer* (- buffer-position 2)))))
                                (unless (and char-1 char-2 (char= char-1 #\\) (char= char-2 #\#)) ; constant
                                   (setf (p-inside-semi-colon-comment-p *parser*) t)))))
                      (34 (calculate-context *parser*) ; quote
                       (setf style-strings-p t)))
                   (dynamically-style-buffer fred style-strings-p (p-inside-semi-colon-comment-p *parser*))
                   (fred-update fred)))))))
              
(defMethod paste :after ((fred  window-fred-item))
   (when (and *do-dynamic-styling* *lisp-file-p*
                 (not (ccl::window-buffer-read-only-p fred))
                 (not (typep fred 'ccl::listener-fred-item)))
      (let ((*rtn-incremental-styling-p* nil)
            (*buffer* (fred-buffer fred)))
        (calculate-context *parser*)
        (dynamically-style-buffer fred t t))
      (fred-update fred)))

(defMethod ed-yank :after ((fred  window-fred-item))
   (when (and *do-dynamic-styling* *lisp-file-p*
                 (not (ccl::window-buffer-read-only-p fred))
                 (not (typep fred 'ccl::listener-fred-item)))
      (let ((*rtn-incremental-styling-p* nil)
            (*buffer* (fred-buffer fred)))
        (calculate-context *parser*)
        (dynamically-style-buffer fred t t))
      (fred-update fred)))

