;;;-*- Mode: Lisp; Package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-prefs.lisp
;;;
;;;      copyright  2008 Glen Foy, all rights reserved,
;;;
;;; ****************************************************************************

(in-package "CC")

(require 'graphic-items "ccl:library;graphic-items")

(defConstant %dialog-width% 530)
(defConstant %dialog-height% 580)
(defConstant %bottom-offset% 3)
(defConstant %max-color-buttons% 27)

;;; constructing these paths dynamically for multiuser systems:
(defun prefs-dir () (ccl::findfolder #$kOnAppropriateDisk #$kPreferencesFolderType))

(defun construct-prefs-path ()
   (merge-pathnames ";org.clairvaux;color-coded;cc-prefs" (prefs-dir)))

(defun construct-style-sets-path ()
   (merge-pathnames ";org.clairvaux;color-coded;style-sets;" (prefs-dir)))

(defun populate-style-sets-folder ()
   (do* ((source-dir (merge-pathnames ";style-sets;" cl-user::*color-coded-pathname*))
          (source-files (directory (merge-pathnames source-dir "*") :files t :directories nil)
                        (rest source-files))
          (source-file (first source-files) (first source-files))
          (source-name (when source-file (pathname-name source-file))
                       (when source-file (pathname-name source-file)))
          (dest-dir (construct-style-sets-path))
          (dest-file (when source-name (merge-pathnames source-name dest-dir))
                     (when source-name (merge-pathnames source-name dest-dir))))
         ((null source-file))
      (unless (probe-file dest-file)
         (copy-file source-file dest-file))))

(defMacro with-cfurl ((cfurl path) &body body)
   `(cond ((probe-file ,path)
            (let ((url-fsref (ccl::make-fsref-from-path ,path)))
              (cond (url-fsref
                      (let ((,cfurl (#_cfurlcreatefromfsref (%null-ptr) url-fsref)))
                        (cond ((%null-ptr-p ,cfurl)
                                (format t "#_CFURLCreateFromFSREF failed.
The complaining macro is WITH-CFURL in color-coded-prefs.lisp."))
                               (t
                                (unwind-protect 
                                   (progn ,@body) 
                                   (#_cfrelease ,cfurl))))))
                     (t
                      (format t (format nil "Could not create a FSREF from: ~S
The complaining macro is WITH-CFURL in color-coded-prefs.lisp." ,path))))))
           (t
            (format t (format nil "The file: ~S does not exist.
The complaining macro is WITH-CFURL in color-coded-prefs.lisp." ,path)))))

(defun open-doc ()
   (let ((file (namestring (merge-pathnames ";color-coded-doc;index.html" 
                                                 cl-user::*color-coded-pathname*))))
     (cond ((probe-file file)
             (with-cfurl (url file)
               (#_lsopencfurlref url (%null-ptr))))
            (t
             (format t (format nil "The documentation file: ~S does not exist." file))))))

(defun small-sys-spec ()
  (if (ccl::module-loaded-p :modern-mcl)
  :small-system-font
  (progn
   (ccl::sys-font-spec)
   (remove :srccopy (font-spec (car ccl::*sys-font-codes*) (- (cdr ccl::*sys-font-codes*) 2))))))

#|
;;; ----------------------------------------------------------------------------
;;;
(defClass PREFS-DIALOG (dialog)
   ((prefs-manager :initform nil :initarg :prefs-manager :reader pd-prefs-manager)))

(defMethod window-save-as ((dialog prefs-dialog))
   (save-as-set (pd-prefs-manager dialog)))

(defMethod window-save ((dialog prefs-dialog))
   (write-prefs (pd-prefs-manager dialog)))
|#

;;; ----------------------------------------------------------------------------
;;;
(defClass PREFS-MANAGER ()
   ((styling-dialog :initform nil :accessor styling-dialog)
    (styling-dialog-items :accessor styling-dialog-items)
    (style-item-spec-dialog-item :accessor style-item-spec-dialog-item)
    (style-item-variable-dialog-item :accessor style-item-variable-dialog-item)
    (font-table-dialog-item :accessor font-table-dialog-item)
    (style-table-dialog-item :initform nil :accessor style-table-dialog-item)
    (fred-dialog-item :accessor fred-dialog-item)
    (fred-doc-dialog-item :accessor fred-doc-dialog-item)
    (spec-button-dialog-item :accessor spec-button-dialog-item)
    (generic-button-dialog-item :accessor generic-button-dialog-item)
    (edit-color-dialog-item :accessor edit-color-dialog-item)
    (color-button-dialog-items :accessor color-button-dialog-items)
    (font-style-dialog-items :accessor font-style-dialog-items)
    (font-size-dialog-items :accessor font-size-dialog-items)
    (case-dialog-items :accessor case-dialog-items)
    (cl-functions-dialog-item :accessor cl-functions-dialog-item)
    (keyword-dialog-item :accessor keyword-dialog-item)
    (exported-symbols-dialog-item :accessor exported-symbols-dialog-item)
    (tab-key-styling-dialog-item :accessor tab-key-styling-dialog-item)
    (dynamic-styling-dialog-item :accessor dynamic-styling-dialog-item)
    (i-beam-dialog-items :accessor i-beam-dialog-items)
    (package-dialog-items :accessor package-dialog-items)
    (style-set-popup-menu :accessor style-set-popup-menu)
    (style-item-variable :accessor style-item-variable)
    (font-color :initform 222 :accessor font-color)
    (old-font-color :initform 777 :accessor old-font-color)
    (palette-color-index :initform 31 :accessor palette-color-index)
    (pref-options :initform *styling-preferences* :accessor pref-options)
    (original-option-values :accessor original-option-values)
    (prefs :initform (sort (copy-list *non-defstyle-prefs*) 'string<) :accessor prefs)
    (original-prefs-values :accessor original-prefs-values)
    (dialog-position :initform (get-centered-dialog-position %dialog-width% %dialog-height%)
                      :accessor dialog-position)
    (dialog-size :initform (make-point %dialog-width% %dialog-height%)
                  :accessor dialog-size)))

(defMethod dump-prefs ((pw prefs-manager))
  (dolist (pref (prefs pw))
     (format t "~%~(~a~)" pref)))
    ; (format t "~%value: ~A" (eval pref))))

;; (dump-prefs *prefs-manager*)

(defMethod add-parameter ((pm prefs-manager) parameter)
   ;; add the new pref keeping the list alphabetized; don't add it twice
   (unless (find parameter (prefs pm))
      (setf (prefs pm) (merge 'list (list parameter) (prefs pm) 'string<)))
   (when (style-table-dialog-item pm)
      (set-table-sequence (style-table-dialog-item pm) 
                          (mapcar #'(lambda (pref)
                                        (string-downcase (string pref)))
                                    (prefs pm)))))

(defun get-var (string)
   (find-symbol (string-upcase string) :cc))

;;; (probe-file (construct-style-sets-path))
;;; (populate-style-sets-folder)

(defMethod open-prefs-dialog ((pm prefs-manager))
   (unless (probe-file (construct-style-sets-path)) (populate-style-sets-folder))
   (save-original-pref-values pm)
   (when (null (styling-dialog pm))
      (setf (styling-dialog pm)
            (make-instance 'dialog
               :window-type
               :movable-dialog
               :window-title
               "Color-Coded"
               :view-nick-name 'prefs-dialog
               :view-position (dialog-position pm)
               :view-size (dialog-size pm)
               :close-box-p nil
               :window-show nil
               :theme-background t
               :prefs-manager pm
               :view-font '("chicago" 12 :srcor :plain (:color-index 0))))
      (setf (styling-dialog-items pm) (get-items pm))
      (apply #'add-subviews (styling-dialog pm) (styling-dialog-items pm)))
   (ccl::deselect-cells (style-table-dialog-item pm))
   (ccl::deselect-cells (font-table-dialog-item pm))
   (cell-select (style-table-dialog-item pm) #@(0 0))
   (scroll-to-cell (style-table-dialog-item pm) #@(0 0))
   (setf (style-item-variable pm) (get-var (cell-contents (style-table-dialog-item pm) #@(0 0))))
   (set-style-options-from-spec pm)
   (update-fields pm)
   (menu-update (style-set-popup-menu pm))
   (let ((*current-package* :common-lisp-user))
     (modal-dialog (styling-dialog pm) nil)))

(defMethod get-items ((pm prefs-manager))
   (append 
    (make-style-item-table pm)
    (make-spec-items pm)
    (make-miscel-items pm)
    (make-style-items pm)
    (make-size-items pm) 
    (make-case-items pm)
    (make-fred-views pm)
    (make-color-button-items pm)
    (make-button-items pm)
    (make-i-beam-radio-buttons pm)
    (make-titles pm)
    (make-check-boxes pm)))

;;; ----------------------------------------------------------------------------
;;; These two classes and some of the support code are borrowed from the 
;;; interface toolkit.
;;;
(defClass STYLE-ITEM (check-box-dialog-item)
   ((attribute :accessor dialog-item-attribute))
   (:documentation "class to support font styles."))

(defMethod initialize-instance :before ((item style-item) &key dialog-item-text)
   (setf (dialog-item-attribute item)
         (find-symbol (string-upcase dialog-item-text) :keyword)))

;;; ----------------------------------------------------------------------------
;;;
(defClass SIZE-ITEM (radio-button-dialog-item)
   ((attribute :accessor dialog-item-attribute))
   (:documentation "class to support font size."))

(defMethod initialize-instance :before ((item size-item) &key dialog-item-text)
   (setf (dialog-item-attribute item)
         (read-from-string dialog-item-text)))

;;; ----------------------------------------------------------------------------
;;;
(defClass CASE-ITEM (radio-button-dialog-item)
   ((attribute :initarg :attribute :accessor dialog-item-attribute))
   (:documentation "class to support font case."))

(defMethod initialize-instance :before ((item case-item) &key dialog-item-text)
   (setf (dialog-item-attribute item)
         (read-from-string dialog-item-text)))

;;; ----------------------------------------------------------------------------
;;;
(defClass PALETTE-BUTTON-ITEM (dialog-item )
   ((color :initform nil :initarg :color :accessor color)
    (selected :initform nil :accessor selected))
   (:documentation "class to support the color palette."))

(defMethod erase-button ((item palette-button-item))
   (without-interrupts
    (with-focused-view item
      (rlet ((rect :rect
                   :topleft #@(0 0)
                   :bottomright (view-size item)))
        (#_eraserect rect)))))

(defMethod view-draw-contents ((item palette-button-item))
   (when (dialog-item-enabled-p item)
      (without-interrupts
       (with-focused-view (view-window item)
         (rlet ((outline :rect
                         :top (point-v (view-position item))
                         :left (point-h (view-position item))
                         :bottom (+ (- (point-v (view-size item)) %bottom-offset%)
                                    (point-v (view-position item)))
                         :right (+ (point-h (view-size item)) (point-h (view-position item)))))
           (with-rgb (color (color item))
             (#_rgbforecolor color)
             (#_paintrect outline))
           (cond ((selected item)
                   (with-rgb (pen-color *black-color*)
                     (#_rgbforecolor pen-color)))
                  (t
                   (with-rgb (pen-color *white-color*)
                     (#_rgbforecolor pen-color))))            
           (#_moveto (point-h (view-position item)) (1- (+ (point-v (view-position item)) 
                                                            (point-v (view-size item)))))
           (#_line (1- (point-h (view-size item))) 0)
           (with-rgb (pen-color *black-color*)
             (#_rgbforecolor pen-color))
           (#_framerect outline))))))

(defMethod make-style-item-table ((pm prefs-manager))
   (list (setf (style-table-dialog-item pm)
              (make-dialog-item 
               'sequence-dialog-item
               #@(5 202) #@(184 96) 
               "fonts"
               #'(lambda (item)
                   (declare (ignore item))
                   (set-style-options-from-spec pm)
                   (update-fields pm))
               :cell-size #@(184 16)
               :view-font (small-sys-spec)
               :selection-type :single
               :table-hscrollp nil
               :table-vscrollp t
               :table-sequence (mapcar #'(lambda (pref)
                                           (string-downcase (string pref)))
                                       (prefs pm))))))

(defMethod make-spec-items ((pm prefs-manager))
   (list 
    (make-dialog-item 'static-text-dialog-item
                      #@(8 3) #@(50 38)
                      "Style:" nil
                      :view-font '("zapfino" 12 :srcor (:color 1445511)))
    (setf (style-item-variable-dialog-item pm) 
          (make-dialog-item 'static-text-dialog-item
                            #@(60 5) #@(279 18)
                            "style item variable" 'nil
                            :view-font '("geneva" 12 :srcor (:color 1445511))))
    (setf (style-item-spec-dialog-item pm)
          (make-dialog-item 'static-text-dialog-item
                            #@(63 23) #@(480 16)
                            "style spec" 'nil
                            :view-font (small-sys-spec)))))

(defMethod set-fred-text ((view scrolling-fred-view) text)
   (set-dialog-item-text view text))

;;; ----------------------------------------------------------------------------
;;;
(defClass BRAIN-DEAD-FRED (fred-item) ())

(defMethod view-draw-contents :after ((f brain-dead-fred))
   (without-interrupts
    (with-focused-view (view-container f)
      (let ((top (1- (point-v (view-position f)) ))
            (left (1- (point-h (view-position f)))))
        (rlet ((rect :rect
                     :top top
                     :left left
                     :bottom (+ 2 top (point-v (view-size f)))
                     :right (+ 2 left (point-h (view-size f)))))
          (with-fore-color *light-gray-color*
            (#_framerect rect)))))))

(defMethod view-key-event-handler ((f brain-dead-fred) char)
   (declare (ignore char))
   ())

(defMethod view-click-event-handler ((f brain-dead-fred) where)
   (declare (ignore where))
   ())

(defMethod make-fred-views ((pm prefs-manager))
   (list (setf (fred-dialog-item pm)
              (make-instance 'scrolling-fred-view
                 :fred-item-class 'brain-dead-fred
                 :view-size #@(414 90)
                 :view-position #@(4 44)
                 :draw-scroller-outline t
                 :h-scrollp nil
                 :margin 5
                 :view-container (styling-dialog pm)))
         (setf (fred-doc-dialog-item pm)
               (make-instance 'scrolling-fred-view
                 :fred-item-class 'brain-dead-fred
                  :view-size #@(186 94)
                  :view-position #@(4 328)
                  :draw-scroller-outline t
                  :h-scrollp nil
                  :margin 2
                  :view-font '("geneva" 9 :srcor :plain (:color 1445511))
                  :view-container (styling-dialog pm)))))

;;; ----------------------------------------------------------------------------
;;;
(defClass STYLE-POP-UP-MENU (pop-up-menu) ())

(defMethod make-miscel-items ((pm prefs-manager))
   (list (setf (font-table-dialog-item pm)
              (make-dialog-item 'sequence-dialog-item
                                #@(316 328) #@(209 93) 
                                "fonts"
                                #'(lambda (item)
                                    (declare (ignore item))
                                    (update-fields pm))
                                :cell-size #@(209 16)
                                :view-font (small-sys-spec)
                                :selection-type :single
                                :table-hscrollp nil
                                :table-sequence 
                                #+ccl-5.2 (sort (remove-if-not 
                                                 #'(lambda (entry)
                                                     (alpha-char-p (elt entry 0)))
                                                 (mapcar #'first *font-list*))
                                                #'string<)
                                #+rmcl (sort 
                                           (remove-if
                                            #'(lambda (entry)
                                                (or
                                                 (not (alpha-char-p (elt entry 0)))
                                                 ;; fonts with these suffixes crash 5.1
                                                 (string= (subseq entry (- (length entry) 2) (length entry))
                                                          "CE")
                                                 (string= (subseq entry (- (length entry) 2) (length entry))
                                                          "CY")))
                                            *font-list*)
                                           #'string<)))
         
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(315 192) #@(210 107) "Font Style:" 'nil 
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(5 146) #@(414 25) 
                           "Color Palette:" 'nil
                           :view-nick-name 'defaults-box
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(195 192) #@(115 107) "Font Size:" 'nil
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(195 444) #@(115 90) "I-Beam:" 'nil
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(195 318) #@(115 104) "Case:" 'nil
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(315 444) #@(210 90) "Modes:" 'nil
                           :view-font (small-sys-spec))
         (make-dialog-item 'ccl::title-box-dialog-item
                           #@(4 444) #@(186 90) "Options:" 'nil
                           :view-font (small-sys-spec))
         
         (make-dialog-item 'static-text-dialog-item
                           #@(432 45) #@(75 18) 
                           "Style Sets:" 'nil
                           :view-font '("optima" 14 :srcor (:color-index 0)))
         
         (setf (style-set-popup-menu pm)
               (make-instance 'style-pop-up-menu
                  :view-size #@(20 20)
                  :view-position #@(504 42)
                  :auto-update-default nil))))


(defMethod make-titles ((pw prefs-manager))
   (list
    (make-dialog-item 'static-text-dialog-item
                      #@(12 184) #@(40 16) 
                      "Styles:" 'nil
                      :view-font (small-sys-spec))

    (make-dialog-item 'static-text-dialog-item
                      #@(12 312) #@(85 16) 
                      "Documentation:" 'nil
                      :view-font (small-sys-spec))

    (make-dialog-item 'static-text-dialog-item
                      #@(328 312) #@(85 14) 
                      "Fonts:" 'nil
                      :view-font (small-sys-spec))))

;;; Use Aqua pop-ups
(setf *use-pop-up-control* t)

(defMethod open-style-set-file ((pm prefs-manager) path)
   (when (read-set pm path (format nil "Color-Coded:

There is a problem with the ~s file.  You should replace it." (pathname-name path)))
      
      (ccl::deselect-cells (style-table-dialog-item pm))
      (ccl::deselect-cells (font-table-dialog-item pm))
      (cell-select (style-table-dialog-item pm) #@(0 0))
      (scroll-to-cell (style-table-dialog-item pm) #@(0 0))
      (setf (style-item-variable pm) (get-var (cell-contents (style-table-dialog-item pm) #@(0 0))))
      (set-style-options-from-spec pm)
      (update-fields pm)))

(defParameter *current-set* "current prefs")

(defMethod menu-update ((menu style-pop-up-menu))
   (let* ((old-items (menu-items menu)))
     (when old-items (apply #'remove-menu-items menu old-items))
     (add-menu-items menu 
                     (make-instance 'menu-item
                        :menu-item-title "current prefs"
                        :menu-item-action  
                        #'(lambda ()
                            (setf *current-set* "current prefs")
                            (open-style-set-file *prefs-manager* (construct-prefs-path)))))
     (add-directory-files menu (construct-style-sets-path))))

(defMethod add-directory-files ((menu menu) dir-path)
   (let* ((files (directory (merge-pathnames  dir-path "*") :files t :directories nil))
          (count 1)
          name)
     (dolist (file files)
        (setf name (pathname-name file))
        (when (string-not-equal name "") ; exclude invisibles 
           (add-menu-items menu (make-instance 'menu-item
                                   :menu-item-title name
                                   :menu-item-action  
                                   (let ((path file)
                                         (name name))
                                     #'(lambda ()
                                         (setf *current-set* name)
                                         (open-style-set-file *prefs-manager* path)))))
           (incf count)))))

(defMethod make-style-items ((pm prefs-manager))
   (labels ((plain-off (item) ; uncheck plain - assumes plain is first
               (let ((checked (check-box-checked-p item)))
                 (when checked
                    (check-box-uncheck (first (font-style-dialog-items pm))))))
            
            (plain-on ()
               (dolist (item (font-style-dialog-items pm))
                  (when (check-box-checked-p item)
                     (return-from plain-on)))
               (check-box-check (first (font-style-dialog-items pm))))
            (button-action (item)
               (plain-off item)
               (plain-on)
               (update-fields pm)))
     (setf (font-style-dialog-items pm)
           (list (make-dialog-item
                  ; choosing plain turns off all others
                  'style-item #@(324 205) #@(72 16) "plain"
                  #'(lambda (item)
                      (let ((checked (check-box-checked-p item)))
                        (dolist (other-item (font-style-dialog-items pm))
                           (unless (eq item other-item)
                              (if checked
                                (check-box-uncheck other-item)))))
                      (plain-on)
                      (update-fields pm))
                  :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
                 (make-dialog-item 'style-item #@(324 227) #@(72 16) "bold"
                                   #'button-action
                                   :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
                 (make-dialog-item 'style-item #@(324 249) #@(72 16) "italic"
                                   #'button-action
                                   :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
                 (make-dialog-item 'style-item #@(411 227) #@(84 17) "condense"
                                   #'button-action
                                   :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
                 (make-dialog-item 'style-item #@(324 271) #@(81 16) "underline"
                                   #'button-action
                                   :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
                 (make-dialog-item 'style-item #@(411 205) #@(72 16) "extend"
                                   #'button-action
                                   :view-font '("geneva" 12 :srcor :plain (:color-index 0)))))))

(defun nil-i-beam-variables ()
   (setf *use-black-i-beam* nil)
   (setf *use-default-i-beam* nil)
   (setf *use-white-i-beam* nil))

(defMethod make-i-beam-radio-buttons ((pm prefs-manager))
   (setf (i-beam-dialog-items pm)
         (list
          (make-dialog-item 'cc::radio-button-dialog-item #@(200 457) #@(70 16) "white"
                            #'(lambda (item)
                                (declare (ignore item))
                                (nil-i-beam-variables)
                                (setf *use-white-i-beam* t)
                                (update-i-beam))
                            :radio-button-cluster 'i-beam-cluster
                            :radio-button-pushed-p *use-white-i-beam*
                            :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
          (make-dialog-item 'cc::radio-button-dialog-item #@(200 479) #@(70 16) "black"
                            #'(lambda (item)
                                (declare (ignore item))
                                (nil-i-beam-variables)
                                (setf *use-black-i-beam* t)
                                (update-i-beam))
                            :radio-button-cluster 'i-beam-cluster
                            :radio-button-pushed-p *use-black-i-beam*
                            :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
          (make-dialog-item 'cc::radio-button-dialog-item #@(200 501) #@(70 16) "default"
                            #'(lambda (item)
                                (declare (ignore item))
                                (nil-i-beam-variables)
                                (setf *use-default-i-beam* t)
                                (update-i-beam))
                            :radio-button-cluster 'i-beam-cluster
                            :radio-button-pushed-p *use-default-i-beam*
                            :view-font '("geneva" 12 :srcor :plain (:color-index 0))))))

(defMethod make-check-boxes ((pm prefs-manager))
   (append (list
              (setf (tab-key-styling-dialog-item pm)
                    (make-dialog-item 'cc::check-box-dialog-item #@(330 457) #@(130 16) 
                                      "indentation styling"
                                      #'(lambda (item)
                                          (declare (ignore item))
                                          (setf *tab-key-styling* (not *tab-key-styling*)))
                                      :check-box-checked-p *tab-key-styling*
                                      :view-font '("geneva" 12 :srcor :plain (:color-index 0))))
              
              (setf (dynamic-styling-dialog-item pm)
                    (make-dialog-item 'cc::check-box-dialog-item #@(330 479) #@(130 16) 
                                      "incremental styling"
                                      #'(lambda (item)
                                          (declare (ignore item))
                                          (setf *do-dynamic-styling* 
                                                (not *do-dynamic-styling*)))
                                      :check-box-checked-p *do-dynamic-styling*
                                      :view-font '("geneva" 12 :srcor :plain (:color-index 0)))))
             (setf (package-dialog-items pm)
                   (list
                    (setf (cl-functions-dialog-item pm)
                          (make-dialog-item 'cc::check-box-dialog-item #@(16 479) #@(100 16) 
                                            "CL functions"
                                            #'(lambda (item)
                                                (declare (ignore item))
                                                (setf *do-cl-package* 
                                                      (not *do-cl-package*))
                                                (update-fields pm))
                                            :check-box-checked-p *do-cl-package*
                                            :view-font '("geneva" 12 :srcor :plain (:color-index 0))))
                    (setf (keyword-dialog-item pm)
                          (make-dialog-item 'cc::check-box-dialog-item #@(16 457) #@(100 16) 
                                            "keywords"
                                            #'(lambda (item)
                                                (declare (ignore item))
                                                (setf *do-keyword-package* 
                                                      (not *do-keyword-package*))
                                                (update-fields pm))
                                            :check-box-checked-p *do-keyword-package*
                                            :view-font '("geneva" 12 :srcor :plain (:color-index 0))))
                    (setf (exported-symbols-dialog-item pm)
                          (make-dialog-item 'cc::check-box-dialog-item #@(16 501) #@(130 16) 
                                            "exported symbols"
                                            #'(lambda (item)
                                                (declare (ignore item))
                                                (setf *do-exported-symbols* 
                                                      (not *do-exported-symbols*))
                                                (update-fields pm))
                                            :check-box-checked-p *do-exported-symbols*
                                            :view-font '("geneva" 12 :srcor :plain (:color-index 0))))))))

(defMethod make-size-items ((pm prefs-manager))
   (labels ((button-action (item)
               (declare (ignore item))
               (update-fields pm)))
     (setf (font-size-dialog-items pm)
           (list
            (make-dialog-item 'size-item #@(202 204) #@(34 16) "9"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(202 226) #@(41 16) "10"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(202 248) #@(40 16) "12"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(202 270) #@(41 16) "14"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(262 204) #@(37 16) "18"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(262 226) #@(42 16) "24"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(262 248) #@(39 16) "36"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'size-item #@(262 270) #@(39 16) "48"
                              #'button-action
                              :radio-button-cluster 'size-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))))))

(defMethod make-case-items ((pm prefs-manager))
   (labels ((button-action (item)
               (declare (ignore item))
               (update-fields pm)))
     (setf (case-dialog-items pm)
           (list
            (make-dialog-item 'case-item #@(202 330) #@(90 16) ":up"
                              #'button-action
                              :attribute :up
                              :view-nick-name 'case-upper                           
                              :radio-button-cluster 'case-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'case-item #@(202 352) #@(90 16) ":down"
                              #'button-action
                              :attribute :down
                              :view-nick-name 'case-lower
                              :radio-button-cluster 'case-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'case-item #@(202 374) #@(50 16) ":cap"
                              #'button-action
                              :attribute :cap3
                              :view-nick-name 'case-capitalize
                              :radio-button-cluster 'case-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'case-item #@(252 374) #@(55 16) ":cap3"
                              #'button-action
                              :attribute :cap3
                              :view-nick-name 'case-capitalize-3
                              :radio-button-cluster 'case-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))
            (make-dialog-item 'case-item #@(202 396) #@(100 16) ":unchanged"
                              #'button-action
                              :attribute :unchanged
                              :view-nick-name 'case-unchanged
                              :radio-button-cluster 'case-cluster
                              :view-font '("geneva" 12 :srcor :plain (:color-index 0)))))))

(defMethod make-button-items ((pm prefs-manager))
   (list (make-dialog-item 'button-dialog-item 
                            #@(450 545) #@(70 20)
                            "Okay" 
                            #'(lambda (item)
                                 (declare (ignore item))
                                 (multiple-value-setq (*ff*  *ms*) 
                                    (font-codes (f-spec '*generic-text-style*)))
                                 (return-from-modal-dialog t))
                            :default-button t)
          (make-dialog-item 'button-dialog-item 
                            #@(10 545) #@(70 20) 
                            "Doc" 
                            #'(lambda (item)
                                 (declare (ignore item))
                                 (open-doc)))
          (make-dialog-item 'button-dialog-item 
                            #@(175 545) #@(70 20) 
                            "Cancel" 
                            #'(lambda (item)
                                 (declare (ignore item))
                                 (restore-original-pref-values pm)
                                 (multiple-value-setq (*ff*  *ms*) 
                                    (font-codes (f-spec '*generic-text-style*)))
                                 (return-from-modal-dialog t)))
          (setf (spec-button-dialog-item pm)
                 (make-dialog-item 'button-dialog-item 
                                   #@(430 75) #@(95 20) 
                                   "Make Spec"
                                   #'(lambda (item)
                                        (declare (ignore item))
                                        (set (style-item-variable pm) *generic-text-style*)
                                        (setf (get (style-item-variable pm) :style-spec)
                                               (s-spec '*generic-text-style*))
                                        (set-style-options-from-spec pm)
                                        (update-fields pm))))
          (setf (generic-button-dialog-item pm)
                 (make-dialog-item 'button-dialog-item 
                                   #@(430 108) #@(95 20) 
                                   "Use Generic"
                                   #'(lambda (item)
                                        (declare (ignore item))
                                        (disable-spec-items pm)
                                        (set (style-item-variable pm) nil)  
                                        (setf (get (style-item-variable pm) :style-spec) nil)
                                        (update-fields pm))))
          (setf (edit-color-dialog-item pm)
                 (make-dialog-item 'button-dialog-item
                                   #@(430 150) #@(95 20)
                                   "Edit Color"
                                   #'(lambda (item)
                                        (declare (ignore item))
                                        (setf (font-color pm) (user-pick-color 
                                                                :color (font-color pm)
                                                                :prompt "pick font color:"))
                                        (update-fields pm))))
          (make-dialog-item 'button-dialog-item 
                            #@(260 545) #@(90 20) 
                            "Save As" 
                            #'(lambda (item)
                                 (declare (ignore item))
                                 (save-as-set pm)))
          (make-dialog-item 'button-dialog-item 
                            #@(365 545) #@(70 20) 
                            "Save" 
                            #'(lambda (item)
                                 (declare (ignore item))
                                 (write-prefs pm)))))

(defMethod deselect-color-buttons ((pm prefs-manager))
   (dolist (button (color-button-dialog-items pm))
      (when (selected button)
         (setf (selected button) nil)
         (view-draw-contents button)
         (return))))

(defMethod make-color-button-items ((pm prefs-manager))
   (let ((items nil)
         (h-coord 15))
     (dotimes (count %max-color-buttons% (setf (color-button-dialog-items pm) 
                                                 (nreverse items)))
        (push (make-dialog-item 'palette-button-item 
                                 (make-point h-coord 157) #@(10 10) 
                                 " " 
                                 #'(lambda (item)
                                     (when (color item) 
                                        (setf (font-color pm) (color item))
                                        (update-fields pm))))
               items)
        (incf h-coord 15))))

(defMethod update-style-item-variable-field ((pm prefs-manager))
   (set-dialog-item-text (style-item-variable-dialog-item pm)
                         (string-downcase (symbol-name (style-item-variable pm))))
   ;; this is to avoid the bogus background coloring of the static-text-item
   (view-draw-contents (style-item-variable-dialog-item pm))
   (validate-view (style-item-variable-dialog-item pm)))

(defMethod update-spec-generic-buttons ((pm prefs-manager))
   (cond ((null (f-spec (style-item-variable pm)))
           (dialog-item-enable (spec-button-dialog-item pm))
           (dialog-item-disable (generic-button-dialog-item pm)))
          (t 
           (cond ((or (eq (style-item-variable pm) '*background-color*)
                        (eq (style-item-variable pm) '*vanilla-styling*)
                        (member (style-item-variable pm) *generic-style-variables*))
                   (dialog-item-disable (generic-button-dialog-item pm)))
                  (t
                   (dialog-item-enable (generic-button-dialog-item pm))))
           (dialog-item-disable (spec-button-dialog-item pm)))))

(defMethod update-color-buttons ((pm prefs-manager))
   (dolist (button (color-button-dialog-items pm))
      (cond ((eq (color button) (font-color pm))
              (when (not (selected button))
                 (setf (selected button) t)
                 (view-draw-contents button)))
             (t 
              (when (selected button)
                 (setf (selected button) nil)
                 (view-draw-contents button))))))

(defMethod update-edit-color-button ((pm prefs-manager))
   ;; If the spec is nil, disable color button; otherwise enable.
   (cond ((null (f-spec (style-item-variable pm)))
           (dialog-item-disable (edit-color-dialog-item pm)))
          (t (dialog-item-enable (edit-color-dialog-item pm)))))

(defMethod update-enable-disable ((pm prefs-manager))
   ;; If the spec is nil, disable the spec items.
   ;; If the spec is not nil, the spec items should be enabled.
   ;; Except for selective disablement below.
   (if (f-spec (style-item-variable pm))
     (enable-spec-items pm)
     (disable-spec-items pm t nil))
   (cond ((eq (style-item-variable pm) '*string-style*)
           (disable-case-items pm))
          ((eq (style-item-variable pm) '*documentation-style*)
           (disable-case-items pm))
          ((eq (style-item-variable pm) 'cc::*background-color*)
           (disable-spec-items pm nil nil))
          ((and (eq (style-item-variable pm) 'cc::*cl-function-style*)
                 (not (check-box-checked-p (cl-functions-dialog-item pm))))
           (disable-spec-items pm t nil))
          ((and (eq (style-item-variable pm) 'cc::*keyword-style*)
                 (not (check-box-checked-p (keyword-dialog-item pm))))
           (disable-spec-items pm t nil))
          ((and (eq (style-item-variable pm) 'cc::*exported-symbol-style*)
                 (not (check-box-checked-p (exported-symbols-dialog-item pm))))
           (disable-spec-items pm t nil))))

(defMethod update-style-item-spec-field ((pm prefs-manager))
   (when (null (f-spec (style-item-variable pm)))
      (cond ((accessor (style-item-variable pm))
              (set-dialog-item-text 
               (style-item-spec-dialog-item  pm)
               (format nil "(using accessor: ~S"
                        (funcall (accessor (style-item-variable pm))))))
             ((eql (var-type (style-item-variable pm)) :def-style)
              (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                    (format nil "(using *generic-def-style*)")))
             ((eql (var-type (style-item-variable pm)) :def-symbol-style)
              (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                    (format nil "(using *generic-def-symbol-style*)")))
             ((member (style-item-variable pm) *cl-function-style-variables*)
              (cond (*do-cl-package*
                      (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                            (format nil "(using *cl-function-style*)")))
                     (t 
                      (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                            (format nil "(using *generic-text-style*)")))))                  
             (t
              (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                    (format nil "(using *generic-text-style*)"))))
      (return-from update-style-item-spec-field))

   (setf (get (style-item-variable pm) :style-spec)
         (append '(:font) (list (construct-font-spec pm))
                   (construct-case-spec pm)))

   (set (style-item-variable pm) (f-pair (s-spec (style-item-variable pm))))
   (without-interrupts
    (cond ((eq (style-item-variable pm) '*background-color*)
            (set-dialog-item-text (style-item-spec-dialog-item  pm)
                                  (format nil "color: ~a" (extract-color-spec pm))))
           (t
            (set-dialog-item-text (style-item-spec-dialog-item pm)
                                  (format nil "~(~s~)" (s-spec (style-item-variable pm))))))
    ;; this is to avoid the bogus background coloring of the static-text-item
    (view-draw-contents (style-item-spec-dialog-item pm))
    (validate-view (style-item-spec-dialog-item pm))))

(defMethod update-sample-text-field ((pm prefs-manager))
   (set-fred-text (fred-dialog-item pm) (sample-code (style-item-variable pm)))
   
   (cond ((eq (style-item-variable pm) '*vanilla-styling*)
           (vanilla-style (fred-dialog-item pm) 
                          0 (buffer-size (fred-buffer (fred-item (fred-dialog-item pm))))))
          (t
           (view-style-buffer (fred-item (fred-dialog-item pm))))))

(defMethod update-sample-text-doc-field ((pm prefs-manager))
   (set-part-color (fred-doc-dialog-item pm) :body *white-color*)
   (set-view-font (fred-doc-dialog-item pm) (list (list :color *black-color*)))
   (set-fred-text (fred-doc-dialog-item pm) (docc (style-item-variable pm)))
   (invalidate-view (fred-doc-dialog-item pm) t))

(defMethod update-i-beam-buttons ((pm prefs-manager))
   (when *use-white-i-beam* (radio-button-push (first (i-beam-dialog-items pm))))
   (when  *use-black-i-beam* (radio-button-push (second (i-beam-dialog-items pm))))
   (when  *use-default-i-beam* (radio-button-push (third (i-beam-dialog-items pm))))
   (update-i-beam))

(defMethod update-package-checkboxes((pm prefs-manager))
   (if *do-cl-package* 
     (check-box-check (first (package-dialog-items pm)))
     (check-box-uncheck (first (package-dialog-items pm))))
   (if *do-keyword-package* 
     (check-box-check (second (package-dialog-items pm)))
     (check-box-uncheck (second (package-dialog-items pm))))
   (if *do-exported-symbols* 
     (check-box-check (third (package-dialog-items pm)))
     (check-box-uncheck (third (package-dialog-items pm)))))

(defMethod update-tab-key-styling ((pm prefs-manager))
   (if *tab-key-styling* 
     (check-box-check (tab-key-styling-dialog-item pm))
     (check-box-uncheck (tab-key-styling-dialog-item pm))))

(defMethod update-dynamic-styling ((pm prefs-manager))
   (cond (*do-dynamic-styling* 
           (check-box-check (dynamic-styling-dialog-item pm)))
          (t
           (check-box-uncheck (dynamic-styling-dialog-item pm)))))

(defMethod update-fields ((pm prefs-manager))
   (update-style-item-variable-field pm)
   (update-style-item-spec-field pm)
   
   (update-package-checkboxes pm)
   (update-enable-disable pm)
   (update-palette pm)
   (update-color-buttons pm)
   
   (update-sample-text-field pm)
   (update-sample-text-doc-field pm)
   (update-spec-generic-buttons pm)
   (update-edit-color-button pm)
   
   (update-i-beam-buttons pm)
   (update-tab-key-styling pm)
   (update-dynamic-styling pm))

(defMethod set-case-from-spec ((pm prefs-manager))
   (let ((case (extract-case-spec pm)))
     (when (listp case) (setq case :cap3)) ; compatibility with an older prefs file format
     (unless (member case '(:up :down :cap :cap3 :unchanged))
        (ed-beep)
        (format t "~%The value for :case in your defstyle form should be either :up, :down :unchanged :cap or :cap3
Your value was: ~S." case)
        (setf case :down))
     (dolist (item (case-dialog-items pm))
        (when (eq case (dialog-item-attribute item))
           (radio-button-push item)))))

(defMethod set-font-from-spec ((pm prefs-manager))
   (with-slots (font-size-dialog-items font-table-dialog-item
                                        font-style-dialog-items)
                pm
      (dolist (item font-style-dialog-items)
         (check-box-uncheck item))      
      (dolist (attr (extract-font-spec pm))
         (cond ((assoc attr *style-alist* :test #'eq)
                 (dolist (item font-style-dialog-items)
                    (when (eq attr (dialog-item-attribute item))
                       (check-box-check item))))
                ((typep attr 'fixnum)
                 (when (zerop attr) (setq attr 12))
                 (dolist (item font-size-dialog-items)
                    (when (eq attr (dialog-item-attribute item))
                       (radio-button-push item))))
                ((typep attr 'string)
                 (when (selected-cells font-table-dialog-item)
                    (dolist (cell (selected-cells font-table-dialog-item))
                       (cell-deselect font-table-dialog-item cell)))
                 (let ((pos (position attr (table-sequence font-table-dialog-item)
                                       :test #'equalp)))
                   (when pos
                      (setq pos (make-point 0 pos))
                      (scroll-to-cell font-table-dialog-item pos)
                      (cell-select font-table-dialog-item pos))))))))

(defMethod set-font-color-from-spec ((pm prefs-manager))
   (setf (font-color pm) (second (first (last (extract-font-spec pm))))))

(defMethod set-style-options-from-spec ((pm prefs-manager))
   (set-style-item-variable pm)
   (set-font-color-from-spec pm)
   (set-case-from-spec pm)
   (set-font-from-spec pm))

(defMethod extract-font-spec ((pm prefs-manager))
   (f-spec (style-item-variable pm)))

(defMethod extract-color-spec ((pm prefs-manager))
   (second (first (last (f-spec (style-item-variable pm))))))

(defMethod extract-case-spec ((pm prefs-manager))
   (c-spec (style-item-variable pm)))

(defMethod construct-font-spec ((pm prefs-manager) &aux temp)
   (append
    (list
     (progn
        (setq temp (car (selected-cells (font-table-dialog-item pm))))
        (if temp
          (cell-contents (font-table-dialog-item pm) temp)
          "chicago")))
    (list (dialog-item-attribute (pushed-radio-button (styling-dialog pm) 'size-cluster)))
    (progn
       (setq temp ())
       (dolist (item (font-style-dialog-items pm) (nreverse temp))
          (when (check-box-checked-p item)
             (push (dialog-item-attribute item) temp))))
    (list (list :color (font-color pm)))))

(defMethod construct-case-spec ((pm prefs-manager))
   (let ((button  (pushed-radio-button (styling-dialog pm) 'case-cluster)))
     (list :case 
          (case (view-nick-name button)
             (case-upper :up)
             (case-lower :down)
             (case-unchanged :unchanged)
             (case-capitalize :cap)
             (case-capitalize-3 :cap3)))))

(defMethod disable-case-items ((pm prefs-manager))
   (dolist (item (case-dialog-items pm))
      (dialog-item-disable item)))

(defMethod disable-spec-items ((pm prefs-manager) 
                                   &optional (disable-color-buttons t)
                                   (unpush-buttons t))
   (dolist (item (font-style-dialog-items pm))
      (dialog-item-disable item)
      (when unpush-buttons (check-box-uncheck item)))
   (dolist (item (font-size-dialog-items pm))
      (dialog-item-disable item)
      (when unpush-buttons (radio-button-unpush item)))
   (dolist (item (case-dialog-items pm))
      (dialog-item-disable item)
      (when unpush-buttons (radio-button-unpush item)))
   (when disable-color-buttons 
      (clear-palette pm)
      (dialog-item-disable (edit-color-dialog-item pm)))
   (dialog-item-disable (font-table-dialog-item pm)))

(defMethod disable-non-color-spec-items ((pm prefs-manager))
   (dolist (item (font-style-dialog-items pm))
      (dialog-item-disable item)
      (check-box-uncheck item))
   (dolist (item (font-size-dialog-items pm))
      (dialog-item-disable item)
      (radio-button-unpush item))
   (dolist (item (case-dialog-items pm))
      (dialog-item-disable item)
      (radio-button-unpush item))
   (dialog-item-disable (font-table-dialog-item pm)))

(defMethod enable-spec-items ((pm prefs-manager))
   (dolist (item (font-style-dialog-items pm))
      (dialog-item-enable item))
   (dolist (item (font-size-dialog-items pm))
      (dialog-item-enable item))
   (dolist (item (case-dialog-items pm))
      (dialog-item-enable item))
   (setf (old-font-color pm) -222)
   (dialog-item-enable (edit-color-dialog-item pm))
   (dialog-item-enable (font-table-dialog-item pm)))

(defun var-type (var)
   (let ((name (symbol-name var)))
     (cond ((and (>= (length name) 4) (string= (subseq name 0 4) "*DEF"))
             (cond ((search "-SYMBOL-STYLE*" name :test 'string=)
                     :def-symbol-style)
                    ((search "-STYLE*" name :test 'string=)
                     :def-style)
                    (t :other)))
            (t :other))))

(defMethod set-defxxxxx-specs-to-nil ((pm prefs-manager))
   (dolist (var (prefs pm))
      (case (var-type var)
        (:def-symbol-style 
          (set var nil))
        (:def-style
          (set var nil)))))

(defMethod save-original-pref-values ((pm prefs-manager))
   ;; note that the values are in reverse order:  
   (setf (original-prefs-values pm) nil)
   (setf (original-option-values pm) nil)
   (dolist (var (prefs pm))
      (push (s-spec var) (original-prefs-values pm)))
   (dolist (var (pref-options pm))
      (push (eval var) (original-option-values pm))))

(defMethod restore-original-pref-values ((pm prefs-manager))
   (let ((values (nreverse (original-prefs-values pm))))
     (dolist (var (prefs pm))
        (setf (get var :style-spec) (pop values))
        (set var (f-pair (s-spec var)))))
   (let ((values (nreverse (original-option-values pm))))
     (dolist (var (pref-options pm))
        (set var (pop values))))
   (update-i-beam))

(defMethod read-prefs ((pm prefs-manager))
   (let ((prefs-path (construct-prefs-path)))
     (when (probe-file prefs-path)
        (unless (read-set pm prefs-path "Color-Coded:

There is a problem with the color-coded preferences file.  It will be deleted, 
and default settings will be used.")
           (delete-file prefs-path)
           (set-defaults pm))
        ;; this is also set after editing with the prefs dialog. 
        ;; it is used in dynamic styling  -- it needs to be set when an image 
        ;; reads the initial style-set.
        (multiple-value-setq (*ff*  *ms*) (font-codes (f-spec '*generic-text-style*))))))

;;; Set the defaults before reading a preference file or a style-set.
;;; This takes care of the case where a prefs file is incomplete or
;;; contains extraneous entries.
;;;
;;; Also called when the prefs file is corrupt.
;;; Could use mapc here, but I like to type ...
(defMethod set-defaults ((pm prefs-manager))
  (do* ((values *non-defstyle-pref-defaults* (rest values))
        (value (first values) (first values))
        (vars *non-defstyle-prefs* (rest vars))
        (var (first vars) (first vars)))
       ((null var))
    (setf (get var :style-spec) value)
    (set var (f-pair (s-spec var))))
  (do* ((values *styling-preferences-defaults* (rest values))
        (value (first values) (first values))
        (vars *styling-preferences* (rest vars))
        (var (first vars) (first vars)))
       ((null var))
    (set var value))
  (do* ((vars  (set-difference (prefs pm) *non-defstyle-prefs*) (rest vars))
        (var (first vars) (first vars))
        (value (when var (default-style var)) (when var (default-style var))))
       ((null var))
     (setf (get var :style-spec) value)
    (set var (f-pair (s-spec var)))))
  
(defun read-prefs-on-startup ()
   (read-prefs *prefs-manager*))

;;; This is used to read the preference file and the styling sets.
;;; The styling sets may contain a subset of (prefs pm).  That's okay because
;;; use-defaults is called first, setting all the variables to defaults or nil.
;;; It is also okay if the preference file has variables that are no longer 
;;; present in (prefs pm).  They are just ignored.
;;;
;;; The reason for this appalling imprecision is that user extensions add new
;;; style-items.  New releases may add or delete style-items. In an effort to
;;; keep preference files usable between release, we will use an a-list here.
;;; If the a-list has the var you are looking for, that's fine.  If not, a default
;;; will be used.  If the a-list has items that are no longer used, they are 
;;; ignored. No harm, no foul.
;;;
(defMethod read-set ((pm prefs-manager) path bad-file-prompt)
   (set-defaults pm)
   (when (probe-file path) 
      (with-open-file (stream path :direction :input)
         (let (a-list entry)
           (setf a-list (read stream nil :eof))
           (when (or (eq a-list :eof) 
                       (not (listp a-list))
                       (not (every 'listp a-list)))
              (open-bad-file-dialog bad-file-prompt)
              (return-from read-set nil))
           (dolist (var *styling-preferences*)
              (setf entry (assoc var a-list))
              (when (consp entry) (set var (second entry))))
           (dolist (var *non-defstyle-prefs*)
              (setf entry (assoc var a-list))
              (when (consp entry) 
                 (setf (get var :style-spec) (second entry))
                 (set var (f-pair (s-spec var)))))
           (dolist (var (prefs pm))
              (setf entry (assoc var a-list))
              (when (consp entry)
                 (setf (get var :style-spec) (second entry))
                 (set var (f-pair (s-spec var)))))
           t))))

(defMethod save-as-set ((pm prefs-manager))
   (let ((set-path (choose-new-file-dialog
                    :directory (construct-style-sets-path)
                    :prompt "Save current configuartion as a styling set."
                    :window-title "Save Set")))
     (when set-path
        (write-prefs pm set-path))))

(defMethod write-prefs ((pm prefs-manager) &optional set-path)
   (let ((path (or set-path (construct-prefs-path))))
     (when (probe-file path) (delete-file path))
     (with-open-file (stream path :direction :output)
        (let (a-list)
          (dolist (var *styling-preferences*)
             (push (list var (eval var)) a-list))
          ;; *** just do (prefs pm)
          (dolist (var *non-defstyle-prefs*)
             (push (list var (get var :style-spec)) a-list))
          (dolist (var (set-difference (prefs pm) *non-defstyle-prefs*))
             (push (list var (get var :style-spec)) a-list))
          (pprint a-list stream)))))

(defun open-bad-file-dialog (prompt)
   
   (let* ((screen-center-width-coord (round (/ *screen-width* 2)))
          (screen-center-height-coord (round (/ *screen-height* 2)))
          (dialog-width 520)
          (dialog-height 150)
          (dialog-x-coord (round (- screen-center-width-coord (/ dialog-width 2))))
          (dialog-y-coord (round (- screen-center-height-coord (/ dialog-height 2))))
          (dialog-position (make-point dialog-x-coord dialog-y-coord))
          (dialog-size (make-point dialog-width dialog-height))
          (bad-prefs-dialog
           
           (make-instance
              'color-dialog
              :window-type
              :double-edge-box
              :window-title
              'nil
              :view-position
              dialog-position
              :window-show nil
              :theme-background t
              :view-size
              dialog-size
              :view-subviews
              (list 
               (make-dialog-item 'static-text-dialog-item #@(5 22) #@(515 70) prompt)
               (make-dialog-item 'button-dialog-item
                                 #@(430 110)
                                 #@(53 20)
                                 "okay"
                                 #'(lambda (item)
                                     (declare (ignore item))
                                     (return-from-modal-dialog nil))
                                 :default-button
                                 t)))))
     (window-show bad-prefs-dialog)
     (when (modal-dialog bad-prefs-dialog t)
        nil)))

(defMethod set-style-item-variable ((pm prefs-manager))
   (setf (style-item-variable pm)
         (get-var (cell-contents (style-table-dialog-item pm)                                        
                                 (first (selected-cells (style-table-dialog-item pm)))))))

(defun update-i-beam ( )
   (cond (*use-white-i-beam* (set-white-i-beam))
          (*use-black-i-beam* (set-black-i-beam))
          (*use-default-i-beam* (set-default-i-beam))))

(defMethod clear-palette ((pm prefs-manager))
   (dolist (button (color-button-dialog-items pm))
      (dialog-item-disable button)
      (setf (color button) nil)
      (erase-button button)))

(defMethod add-new-color-to-palette-buttons ((pm prefs-manager) new-color)
   (with-slots (color-button-dialog-items) pm
      (let ((insertion-index 0))
        (dolist (button color-button-dialog-items)
           (cond ((color button) 
                   (incf insertion-index)
                   (when (>= insertion-index %max-color-buttons%)
                      (return-from add-new-color-to-palette-buttons)))
                  (t (let ((button (nth insertion-index color-button-dialog-items)))
                       (setf (color button) new-color)
                       (dialog-item-enable button)
                       (view-draw-contents button)
                       (return))))))))

(defMethod update-palette ((pm prefs-manager))
   ;; Check for a change in the color palette, and rebuild if necessary.
   ;; when the spec is nil, the palette should be clear
   (when (or (null (eval (style-item-variable pm)))
               (and (eq (style-item-variable pm) 'cc::*cl-function-style*)
                     (not (check-box-checked-p (cl-functions-dialog-item pm))))
               (and (eq (style-item-variable pm) 'cc::*keyword-style*)
                     (not (check-box-checked-p (keyword-dialog-item pm)))))
      (clear-palette pm)
      (return-from update-palette))
   (when (equal (font-color pm) (old-font-color pm)) (return-from update-palette))
   (setf (old-font-color pm) (font-color pm))
   (let ((new-color-set nil)
         (old-color-set nil))
     (dolist (button (color-button-dialog-items pm))
        (when (not (null (color button))) (push (color button) old-color-set)))
     (dolist (var (prefs pm))
        (when (listp (eval var))
           (let* ((font-spec (f-spec var))
                  (color (second (first (last font-spec)))))
             (when color 
                (pushnew color new-color-set :test #'=)))))
     (cond ((and (subsetp old-color-set new-color-set) 
                   (subsetp new-color-set old-color-set))
             (return-from update-palette))
            (t (clear-palette pm) 
               (dolist (color new-color-set)
                  (add-new-color-to-palette-buttons pm color))))))

(setf *prefs-manager* (make-instance 'prefs-manager))

(defun open-style-prefs-dialog ()
   (open-prefs-dialog *prefs-manager*))


