;;;-*- Mode: Lisp; Package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-defstyle.lisp
;;;      
;;;      copyright  2008 Glen Foy, all rights reserved,
;;;
;;;      Defstyle and calls to defsyle.
;;;
;;; ****************************************************************************

(in-package "CC")

;;; DEFSTYLE is used to define the built-in stylable forms and also to extend the utility 
;;; so that it will style homegrown macros. User extensions are integrated into 
;;; the preference dialog and are identical in every way to the built-ins.  See the 
;;; documentation for examples of user extensions. The prototype of this macro was 
;;; contributed by Octav Popescu.
;;;
;;; Note that DEFSTYLE is not exported.  Put your extensions at the end of 
;;; this file.
;;;  
(defMacro defstyle (name description &key (add-to-hash-table-p t)
                            default-style default-symbol-style 
                            doc sample-code inherits-style inherits-symbol-style
                            style-accessor symbol-style-accessor)
   "Both built-in and user-defined styles are created by calls to this macro.User extensions are inserted 
in the prefs dialog where they can be edited and saved in the usual manner."
   (let* ((style-name (intern (concatenate 'string (string-upcase (string name)) "-STYLE") :cc))
          (symbol-style-name (intern (concatenate 'string (string-upcase (string name)) "-SYMBOL-STYLE") :cc))
          (parameter-style-name (intern (concatenate 'string "*" (string style-name) "*") :cc))
          (parameter-symbol-style-name (intern (concatenate 'string "*" (string symbol-style-name) "*") :cc)))
     (flet ((define-forms (param-name func-name default-style accessor inherits-style)
              (cond ((and doc sample-code)
                      `(progn
                          (defParameter ,param-name (when ,default-style (f-pair ,default-style)))
                          (setf (get ',param-name :default-style) ,default-style)
                          (setf (get ',param-name :doc) ,doc)
                          (setf (get ',param-name :sample-code) ,sample-code)
                          (setf (get ',param-name :accessor) ,accessor)
                          (setf (get ',param-name :style-spec) ,default-style)
                          (add-parameter *prefs-manager* ',param-name)
                          ,(cond (accessor
                                   `(defun ,func-name () (if ,param-name
                                                              ,param-name
                                                              (or (funcall ,accessor)
                                                                   *generic-text-style*))))
                                  (t
                                   `(defun ,func-name () ,(if inherits-style
                                                               `(or ,param-name ,inherits-style)
                                                               `(or ,param-name *generic-text-style*)))))))
                     (t ; don't add to dialog
                      (cond (accessor
                              `(progn
                                  (defParameter ,param-name (when ,default-style (f-pair ,default-style)))
                                  (setf (get ',param-name :default-style) ,default-style)
                                  (setf (get ',param-name :style-spec) ,default-style)
                                  (defun ,func-name () (if ,param-name
                                                            ,param-name
                                                            (or (funcall ,accessor)
                                                                 *generic-text-style*)))))
                             (t
                              `(progn
                                  (defParameter ,param-name (when ,default-style (f-pair ,default-style)))
                                  (setf (get ',param-name :default-style) ,default-style)
                                  (setf (get ',param-name :style-spec) ,default-style)
                                  (defun ,func-name () ,(if inherits-style
                                                             `(or ,param-name ,inherits-style)
                                                             `(or ,param-name *generic-text-style*))))))))))
       `(progn
           ;; static and incremental parsing techniques:
           (construct-function ,name ',description ,add-to-hash-table-p)
           (construct-network ,name ',description)
           ,(when (member 'macro description)
               (define-forms parameter-style-name style-name default-style 
                 style-accessor inherits-style))
           ,(when (or (member 'symbol description)
                        (member 'struct-sym description))
               (define-forms parameter-symbol-style-name symbol-style-name 
                 default-symbol-style symbol-style-accessor inherits-symbol-style))))))

;;; ----------------------------------------------------------------------------
;;; Built-ins and user extensions are both defined by a general mechanism.
;;; Below are the defstyle calls for the builtin stylable forms.
;;;
(defStyle "DEFPACKAGE" (superparen macro symbol body superparen)
  :default-symbol-style '(:font ("monaco" 9 :underline (:color 1445511)) :case :down)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defpackage \"Color-Coded\" (:use :cl :ccl) (:nicknames :cc))"
  :doc " You can give this a unique style or
click the use generic button to give 
it the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFPARAMETER" (superparen macro symbol form doc superparen)
  :default-symbol-style '(:font ("monaco" 9 :plain (:color 1445511)) :case :down)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defparameter *squash* 'zucchini \"Add onions and some butter\")"
  :doc " You can give this a unique style or
click the use generic button to give 
it the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFVAR" (superparen macro symbol form doc superparen)
  :default-symbol-style '(:font ("monaco" 9 :plain (:color 1445511)) :case :down)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defvar *vegetable* 'eggplant \"Casserole with tomatos and mozzarella\")"
  :doc " You can give this a unique style or
click the use generic button to give 
it the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFCONSTANT" (superparen macro symbol form doc superparen)
  :default-symbol-style '(:font ("monaco" 9 :plain (:color 1445511)) :case :down)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defconstant *roast* 'eye-of-round \"Slow cook with onions, carrots, mushrooms\")"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFCLASS" (superparen macro symbol derivation-list slot-list options superparen)
   :default-style '(:font ("monaco" 9 :plain (:color 0)) :case (:capitalize 3))
   :default-symbol-style '(:font ("monaco" 9 :underline (:color 1445511)) :case :up)
   :inherits-style *generic-def-style*
   :inherits-symbol-style *generic-def-symbol-style*
   :sample-code "(defClass BACON-LETTUCE-AND-TOMATO (sandwich)
    ((bacon :initform 'hickory-smoked :accessor bit-bacon)
    (lettuce :initform 'romano :accessor blt-lettuce)
    (tomato :initform 'ripe :accessor blt-tomato))
    (:documentation \"Bread is innerited-from-sandwidh\"))"
   :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFUN" (superparen macro symbol parameter-list doc body superparen)
  :default-style '(:font ("monaco" 9 :plain (:color 0)) :case :down)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defun slice-bread (&optional (loaf \"pumpernickle\"))
   \"Character-sized slices\"
   (coerce loaf 'list))"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFMACRO" (superparen macro symbol parameter-list doc body superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defmacro make-sandwich (ingredient-1 ingredient-2 ingredient-3)
   \"Assemble ingredients between slices of bread.\"
   `(slice-of-bread ,ingredient-1 ,ingredient-2 ,ingredient-3 slice-of-bread))"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFINE-COMPILER-MACRO" (superparen macro symbol parameter-list doc body superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(define-compiler-macro apply  (&whole call &environment env fn arg0 &rest args)
  (let ((original-fn fn))
    (if (and arg0 
             (null args)
             (consp fn)
             (eq (%car fn) 'function)
             (null (cdr (%cdr fn)))
             (consp (setq fn (%cadr fn)))
             (eq (%car fn) 'lambda))
      (destructuring-bind (lambda-list &body body) (%cdr fn)
        `(destructuring-bind ,lambda-list ,arg0 ,@body))
      (let ((last (%car (last (push arg0 args)))))
        (if (and (consp last) (memq (%car last) '(cons list* list)))
          (cons (if (eq (%car last) 'list) 'funcall 'apply)
                (cons
                 original-fn
                 (nreconc (cdr (reverse args)) (%cdr last))))
          call)))))"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFINE-MODIFY-MACRO" (superparen macro symbol parameter-list form doc superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(define-modify-macro appendf (&rest args) 
    append \"Append onto list\")"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFINE-SETF-EXPANDER" (superparen macro symbol parameter-list doc body superparen)
   :inherits-style *generic-def-style*
   :inherits-symbol-style *generic-def-symbol-style*
   :sample-code "(define-setf-expander lastguy (x &environment env)
   \"Set the last element in a list to the given value.\"
   (multiple-value-bind (dummies vals newval setter getter)
       (get-setf-expansion x env)
     (let ((store (gensym)))
       (values dummies
               vals
               `(,store)
               `(progn (rplaca (last ,getter) ,store) ,store)
               `(lastguy ,getter)))))"
   :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFINE-CONDITION" (superparen macro symbol derivation-list slot-list options superparen)
   :inherits-style *generic-def-style*
   :inherits-symbol-style *generic-def-symbol-style*
   :sample-code "(define-condition tsm-get-event-parameter-error (tsm-error)
    ((error-code :initarg :error-code :reader tsm-error-code)
     (parameter-name :initarg :parameter-name :reader tsm-error-parameter-name)))"
   :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")
                                   
(defStyle "DEFGENERIC" (superparen macro symbol parameter-list options superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defgeneric make-supper (main-course)
   (:documentation \"Supper depends on what's in the frig.\"))"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "DEFMETHOD" (superparen macro symbol qualifier parameter-list doc body superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defmethod make-supper ((roast roast-main-course))
   \"Cook for 3 hours with veggies.\"
   (let ((pot nil))
     (push roast pot)
     (push 'onions pot)
     (push 'carrots pot)
     (slow-cook pot 3)))"
  :doc " You can give this a unique style or
 click the use default button to give it 
the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

(defStyle "LABELS" (paren macro embedded-function-definitions body paren)
   :style-accessor 'cl-package-style
   :sample-code "(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (labels ((add-ingredient (ingredient)
             \"Add ingredient and stir.\"
             (push ingredient pot)
             (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))"
   :doc " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*cl-function-style*.")

(defStyle "LAMBDA" (superparen macro parameter-list doc body superparen)
   :style-accessor 'cl-package-style
   :sample-code "(defun make-soup-recipe (chicken onions carrots)
  \"Return the function for making soup.\"
   #'(lambda (chicken onions carrots)
       \"Don't forget to stir.\"
        (flet ((add-ingredient (ingredient)
                 \"Add ingredient and stir.\"
                 (push ingredient pot)
                 (stir-pot pot))
               (simmer ( minutes pot)
                 \"Keep the temp low.\"
                 (move-to-back-burner pot)
                 (set-temperature-to-low)
                 (set-timer minutes)
                 (cook pot)))
        (add-ingredient chicken)
        (simmer 120 pot)
        (add-ingredient onions)
        (simmer 30 pot)
        (add-ingredient carrots)
        (simmer 30 pot))))"
   :doc " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
the accessor function.")

(defStyle "FLET" (paren macro embedded-function-definitions body paren)
   :style-accessor 'cl-package-style
   :sample-code "(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (flet ((add-ingredient (ingredient)
           \"Add ingredient and stir.\"
           (push ingredient pot)
           (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))"
   :doc " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
the accessor function.")

(defStyle "LOOP" (loop-superparen macro loop-body loop-superparen)
   :style-accessor 'cl-package-style
   :sample-code "(defun name-clash (accessor-name superclasses env)
  (loop for superclass in superclasses 
        for sd = (find-sub-sd superclass env)
        thereis (and sd
                     (find accessor-name (sd-refnames sd)))))"
   :doc " You can give this a unique style 
or click the use generic button to 
give it the style defined by 
the accessor function.")

(defStyle "DEFSTRUCT" (superparen macro struct-sym doc struct-fields superparen)
  :default-symbol-style '(:font ("monaco" 9 :underline (:color 1445511)) :case :up)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*
  :sample-code "(defstruct italian-meal
   \"Garlic is good for you.\"
   appetizer
   (pasta 'fettuchine)
   (main-course 'veal-parmesan)
   salad)"
  :doc " You can give this a unique style or
click the use generic button to give 
it the generic style defined by 
*generic-def-symbol-style*.")

;;; This is just used by the prefs dialog.
(defStyle "DEFXXXXX" (superparen macro symbol qualifier parameter-list doc body superparen)
  :inherits-style *generic-def-style*
  :inherits-symbol-style *generic-def-symbol-style*)

(defStyle "DEFSTYLE" (superparen macro symbol variable-definitions keyword-pairs superparen)
   :default-style '(:font ("monaco" 9 :plain (:color 0)) :case (:capitalize 3))
   :default-symbol-style '(:font ("monaco" 9 :underline (:color 1445511)) :case :down)
   :inherits-style *generic-def-style*
   :inherits-symbol-style *generic-def-symbol-style*
   :sample-code "(defstyle \"LABELS\" (style-paren style-macro-name style-embedded-function-definitions
     style-body style-paren)
  :inherits-style *cl-function-style*
  :sample-code \"(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (labels ((add-ingredient (ingredient)
             \"Add ingredient and stir.\"
             (push ingredient pot)
             (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))\"
  :doc \" You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*cl-function-style*.\")"
   :doc " You can give this a unique style or
click the use generic button to give 
it the generic style defined by 
*generic-def-style* or
*generic-def-symbol-style*.")

;;; These five are special cases.  They are not the names of forms, but 
;;; rather special syntax within stylable forms.  They appear as prefs in
;;; the prefs dialog, but are not put in the defstyle hash-table.
;;; The :add-to-hash-table-p nil keyword arg suppresses the addition of the 
;;; function to the hash-table.
(defStyle "EMBEDDED-FUNCTION" (superparen symbol parameter-list doc body superparen)
   :add-to-hash-table-p nil
   :inherits-style *generic-text-style*
   :default-symbol-style '(:font ("monaco" 9 :underline (:color 1445511)) :case :down)
   :sample-code "(defun make-soup (pot carrots onions chicken)
  \"Cook for 3 hours with veggies.\"
  (flet ((add-ingredient (ingredient)
           \"Add ingredient and stir.\"
           (push ingredient pot)
           (stir-pot pot))
           (simmer ( minutes pot)
             \"Keep the temp low.\"
             (move-to-back-burner pot)
             (set-temperature-to-low)
             (set-timer minutes)
             (cook pot)))
    (add-ingredient chicken)
    (simmer 120 pot)
    (add-ingredient onions)
    (simmer 30 pot)
    (add-ingredient carrots)
    (simmer 30 pot)))"
   :doc " The style of the names of functions
 created with labels or flet.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(defStyle "VARIABLE-DEFINITION" (optional-paren symbol body optional-paren)
   :add-to-hash-table-p nil
  :inherits-style *generic-text-style* 
    :default-symbol-style '(:font ("monaco" 9 :italic (:color 0)) :case :down)
    :sample-code "(defmethod make-pasta ((fettuccine pasta) &optional (veggie 'broccolli) 
                                                       (sauce 'white-sauce))
   \"Don't over-cook the white sauce.\"
   (let ((plate nil))
      (push (cook-al-dente fettuccine) plate)
      (push (lightly-steamed veggie) plate)
      (pour sauce plate)))"
    :doc " The style of variables bound in do, 
let, let*, etc.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

(defStyle "CASE-MATCH" (optional-paren symbol body optional-paren)
   :add-to-hash-table-p nil
  :inherits-style *generic-text-style* 
    :default-symbol-style '(:font ("monaco" 9 :italic (:color 0)) :case :down)
    :sample-code "(defun pizza-ingredients (your-current-weight)
   (case your-current-weight
     (skinny '(extra-cheese sausage mushrooms))
     (average '(cheese sausage mushrooms))
     (a-little-chubby '(cheese))
     (t '(lettuce))))"
    :doc " The style of case statement
match constants.

You can give this a unique style 
or click the use generic button to 
give it the style defined by 
*generic-text-style*.")

;;; Struct symbols require their own network/styling function, since they
;;; can be a symbol or a list of symbol and options.  This is not put in
;;; the prefs dialog and not put in the hash-table.  The function struct-sym
;;; (color-coded-styling.lisp) calls this directly.
;;;
(defStyle "STRUCT-SYM" (optional-paren symbol ancestor body optional-paren)
   :add-to-hash-table-p nil
   :symbol-style-accessor 'defstruct-symbol-style)

;;; These are special cases.  They do not appear in the prefs dialog.  The 
;;; macros always receive *cl-package-style*, or if that is nil, they receive
;;; *generic-text-style*.  They are defined in order to correctly style the 
;;; variable sub-form.  Without this, variables which have names found in :cl will be
;;; incorrectly styled as :cl syntax. 
;;;
;;; They have no doc or sample-code. Defstyle doesn't create params for calls 
;;; that do not have doc and sample-code, and they are not inserted in the dialog's list.
;;;
(defStyle "DOTIMES" (paren macro variable-form body paren)
  :style-accessor 'cl-package-style)

(defStyle "DOLIST" (paren macro variable-form body paren)
  :style-accessor 'cl-package-style)

(defStyle "MULTIPLE-VALUE-BIND" (paren macro variable-list body paren)
  :style-accessor 'cl-package-style)

(defStyle "MULTIPLE-VALUE-SETQ" (paren macro variable-list body paren)
  :style-accessor 'cl-package-style)

(defStyle "DESTRUCTURING-BIND" (paren macro parameter-list body paren)
  :style-accessor 'cl-package-style)

(defStyle "DO" (paren macro variable-definitions form body paren)
  :style-accessor 'cl-package-style)

(defStyle "DO*" (paren macro variable-definitions form body paren)
  :style-accessor 'cl-package-style)

(defStyle "LET" (paren macro variable-definitions body paren)
  :style-accessor 'cl-package-style)

(defStyle "LET*" (paren macro variable-definitions body paren)
  :style-accessor 'cl-package-style)

(defStyle "PROG" (paren macro variable-definitions body paren)
  :style-accessor 'cl-package-style)

(defStyle "PROG*" (paren macro variable-definitions body paren)
  :style-accessor 'cl-package-style)

(defStyle "WITH-SLOTS" (paren macro variable-definitions form body paren)
  :style-accessor 'cl-package-style)

(defStyle "WITH-ACCESSORS" (paren macro variable-definitions form body paren)
  :style-accessor 'cl-package-style)

(defStyle "WITH-OPEN-FILE" (paren macro variable-form body paren)
   :style-accessor 'cl-package-style)

;;; These are all special cases:

;;; This is defined here to correctly handle the embedded-function (macro) definitions.
;;; There is an entry in the prefs dialog for embedded-function-definitions.
(defStyle "MACROLET" (paren macro embedded-function-definitions body paren)
  :style-accessor 'cl-package-style)

;;; defined to correctly style the match-forms.
;;; There is an entry in the prefs dialog for case-match-forms.
(defStyle "CASE" (paren macro form case-match-forms paren)
  :style-accessor 'cl-package-style)

(defStyle "CCASE" (paren macro form case-match-forms paren)
   :style-accessor 'cl-package-style)

(defStyle "ECASE" (paren macro form case-match-forms paren)
  :style-accessor 'cl-package-style)

(defStyle "TYPECASE" (paren macro form case-match-forms paren)
  :style-accessor 'cl-package-style)

(defStyle "ETYPECASE" (paren macro form case-match-forms paren)
  :style-accessor 'cl-package-style)

(defStyle "CTYPECASE" (paren macro form case-match-forms paren)
  :style-accessor 'cl-package-style)

;;; Style the unique superparens.
;;; There is an entry in the prefs dialog for eval-when-superparen.
(defStyle "EVAL-WHEN" (eval-when-superparen macro form body eval-when-superparen)
  :style-accessor 'cl-package-style)

;;; ----------------------------------------------------------------------------
;;; USER EXTENSIONS GO HERE:
;;; ----------------------------------------------------------------------------

#|
(defStyle "while" (paren macro form body paren)
  :style-accessor 'cl-package-style)

(defStyle "WHILE" (paren macro form body paren)
   :style-accessor 'cl-package-style
   :sample-code "(defun foo (stop)
  (let ((count 0))
    (while (< count stop)
      (format t \"~%Foo\")
      (incf count))))"
   :doc " You can give this a unique style
or click the use generic button to
give it the style defined by
the accessor function.")
|#







;;; ----------------------------------------------------------------------------
;;; NOTE: extensions must come before this call
;;; ----------------------------------------------------------------------------
;;;
(read-prefs *prefs-manager*)

(provide "COLOR-CODED")

