;;;-*- Mode: Lisp; Package: (COLOR-CODED (CL CCL)) -*-

;;; ****************************************************************************
;;; 
;;;      color-coded-comments.lisp
;;;
;;;      copyright  2008 Glen Foy, all rights reserved,
;;;
;;;     These classes support the styling of semi-colon and sharp-stroke comments,
;;;     and strings.  Most unusual cases are correctly handled: strings embedded in 
;;;     comments, comments inside of strings, etc.  There are a few pathological 
;;;     combinations which will result in incorrect styling.
;;; 
;;; ****************************************************************************

(in-package "CC")

;;; ----------------------------------------------------------------------------
;;;
(defClass STYLED-COMMENT ()
  ((buffer :initarg :comment-buffer :reader comment-buffer)
   (comment-start :initarg :comment-start :initform nil :reader comment-start)
   (comment-end :initform nil :initarg :comment-end :reader comment-end))
  (:documentation "support for styled comments."))

;;; ----------------------------------------------------------------------------
;;;
(defClass STYLED-SEMI-COLON-COMMENT (styled-comment) ())

;;; ----------------------------------------------------------------------------
;;;
(defClass STYLED-SHARP-COMMENT (styled-comment) ())

(defMethod style-comment ((comment styled-semi-colon-comment))
  (let* ((code-list (fcodes *semi-colon-comment-style*))
         (ff (first code-list))
         (ms (second code-list)))
    (buffer-set-font-codes (comment-buffer comment) ff ms 
                           (comment-start comment) (comment-end comment)))
  t)

(defMethod style-comment ((comment styled-sharp-comment))
  (let* ((code-list (fcodes *sharp-comment-style*))
         (ff (first code-list))
         (ms (second code-list)))
    (buffer-set-font-codes (comment-buffer comment) ff ms 
                           (comment-start comment) (comment-end comment))))

;;; ----------------------------------------------------------------------------
;;;
(defClass STYLED-STRING ()
  ((buffer :initarg :string-buffer :reader string-buffer)
   (string-start :initarg :string-start :initform nil :reader string-start)
   (string-end :initform nil :initarg :string-end :reader string-end))
  (:documentation "support for styled strings."))

(defMethod style-string ((string styled-string))
  (let* ((code-list (fcodes *string-style*))
         (ff (first code-list))
         (ms (second code-list)))
    (cond (*dynamic-p* ; if dynamic, never style past *dynamic-pos* 
           (buffer-set-font-codes (string-buffer string) ff ms 
                                  (string-start string) *dynamic-pos*))
          (t
           (buffer-set-font-codes (string-buffer string) ff ms 
                                  (string-start string) (string-end string))))))

;;; ----------------------------------------------------------------------------
;;;
(defClass SEGMENT-ARRAY ()
  ((array :initarg :array :reader segment-array-array)
   (length :initarg :length :accessor segment-array-length))
  (:documentation 
   "a sorted 2d array of the start and end positions for segments  in
a buffer.  there are three segment types: strings, semi-colon comments, 
and sharp-stroke comments.  the method not-embedded-in-segment-p does
 a binary search for the position of a particular char to see if the 
char is embedded."))

(defun unify-segment-lists (segment-list-1 segment-list-2)
  "Merge two lists, discarding segments which are embedded in segments of the other list."
  (do* ((list-1 segment-list-1)
        (list-2 segment-list-2)
        (segment-1 (first list-1) (first list-1))
        (segment-2 (first list-2) (first list-2))
        (unified-list nil))
       ((and (endp list-1) (endp list-2)) (nreverse unified-list))
    (cond ((and list-1 list-2)
           (cond ((< (first segment-1) (first segment-2))
                  (cond ((< (first segment-2) (second segment-1))
                         (pop list-2))
                        (t 
                         (push segment-1 unified-list)
                         (pop list-1))))
                 (t
                  (cond ((< (first segment-1) (second segment-2))
                         (pop list-1))
                        (t 
                         (push segment-2 unified-list)
                         (pop list-2))))))
          (t ; one list is empty - add what's left of the other
           (cond ((endp list-1)
                  (return (append (nreverse unified-list) list-2)))
                 (t
                  (return (append (nreverse unified-list) list-1))))))))

(defun make-segment-array (table)
  "Constructor for the segment-array class."
  (let ((table-length (length table)))
    (make-instance 'segment-array
      :length table-length
      :array (make-array `(,table-length 2)
                         :initial-contents table))))

;;; This is called when constructing the segment array and to get a list of strings
;;; to style. When styling dynamically, cull the string list. When constructing the 
;;; segment array, don't.
;;;
(defun create-string-list (buffer start end  &optional styling-p)
  "Return a list of the form, (start end), for each string in buffer."
  (flet ((semi-colon-commented-p (pos)
           (do* ((start (buffer-line-start buffer pos) (1+ start))
                 (char (buffer-char buffer start) (buffer-char buffer start)))
                ((>= start pos))
             (when (char= char #\;) (return-from semi-colon-commented-p t))))
         (sharp-stroke-commented-p (pos)
           (do ((start pos (1- start))
                (char (buffer-char buffer start) (buffer-char buffer start))
                (char-minus-one 
                 (when (>= start 1) (buffer-char buffer (1- pos)))
                 (when (>= start 1) (buffer-char buffer (1- pos)))))
               ((or (= start 1)
                    (and (char= char #\#) (char= char-minus-one #\|))))
             (when (and (char= char #\|) 
                        (char= char-minus-one #\|))
               (return-from sharp-stroke-commented-p t)))))
    (do* ((position start)
          (string-list nil)
          string-end)
         ((> position end) (nreverse string-list))
      (cond ((and (eql (buffer-char buffer position) #\") 
                  (not (eql (buffer-char buffer (cond ((> position 0)
                                                       (- position 1))
                                                      (t  position))) #\\))
                  ;; Too expensive; may have a rare mis-styled file
                  ;; because of an unmatched quote in a sharp-comment - oh well ...
                  ;; (not (sharp-stroke-commented-p position))
                  (not (semi-colon-commented-p position)))
             (setf string-end (buffer-fwd-sexp buffer position))
             (cond ((and string-end (<= string-end end))
                    ;; Support for dynamic styling - only cull the string list
                    ;; when styling strings, not when constructing the segment array
                    (if *dynamic-p* 
                      (if styling-p
                        ;; cull
                        (when (and (>= *dynamic-pos* position)
                                   (<= *dynamic-pos* string-end))
                          (push (list position string-end) string-list))
                        (push (list position string-end) string-list))
                      (push (list position string-end) string-list))
                    (setf position string-end))
                   (t 
                    (return (nreverse string-list)))))
            (t 
             (incf position))))))

(declaim (inline line-end))

(defun line-end (buffer position)
   ;;; LineFeeds on a Mac - arrrgh!!
   (let ((newline-pos (ccl::buffer-forward-search buffer #\newline position))
         (linefeed-pos (ccl::buffer-forward-search buffer #\linefeed position)))
     (cond ((and newline-pos linefeed-pos)
             (min newline-pos linefeed-pos))
            (newline-pos newline-pos)
            (linefeed-pos linefeed-pos)
            (t nil))))

;;; This is only called by get-combined-segment-list, when doing vanilla styling.
(defun create-semi-colon-comment-list (buffer start end )
   "Return a list of the form, (start end), for each comment in buffer."
   (do* ((position start)
          (comment-list nil)
          comment-end)
         ((> position end) (nreverse comment-list))
      (cond ((and (eql (buffer-char buffer position) #\;) 
                    (> position 0) ; mode line ???
                    (not (eql (buffer-char buffer (- position 1)) #\\)))
              (setf comment-end (line-end buffer position))
              (cond ((and comment-end (<= comment-end end))
                      (push (list position (1- comment-end)) comment-list)
                      (setf position (1+ comment-end)))
                     (t ; hum ...
                      (incf position))))
             (t
              (incf position)))))

;;; This is only called by get-combined-segment-list, when doing vanilla styling.
(defun create-sharp-stroke-comment-list (buffer start end )
  "Return a list of the form, (start end), for each comment in buffer."
  (do* ((position start)
        (comment-list nil)
        comment-end)
       ((> position end) (nreverse comment-list))
    (cond ((and (eql (buffer-char buffer position) #\#)
                (eql (buffer-char buffer (1+ position)) #\|)
                (> position 0)
                (not (eql (buffer-char buffer (- position 1)) #\\)))
           (setf comment-end (buffer-fwd-sexp buffer position))
           (cond ((and comment-end (<= comment-end end))
                  (push (list position comment-end) comment-list)
                  (setf position (1+ comment-end)))
                 (t 
                  (return (nreverse comment-list)))))
          (t
           (incf position)))))

(defMethod not-embedded-in-segment-p ((array segment-array) position)
  ;; Do a binary search of the segment-array to see if the position is embedded.
  (when (zerop (segment-array-length array)) (return-from not-embedded-in-segment-p t))
  (when (null position) (return-from not-embedded-in-segment-p nil))
  (do* ((top (1- (segment-array-length array)))
        (bottom 0)
        (index (truncate (+ bottom top) 2) (truncate (+ bottom top) 2)))
       ((< top bottom) t)
    (when (and (< (aref (segment-array-array array) index 0) position)
               (> (aref (segment-array-array array) index 1) position))
      ;; embedded
      (return nil))
    (cond ((<= position (aref (segment-array-array array) index 0))
           (setf top (1- index)))
          ((>= position (aref (segment-array-array array) index 1))
           (setf bottom (1+ index)))
          (t (error "~&Bad value in binary search: ~a" position)))))

(defun embedded-in-segment-p (pos)
  (when *segment-array*
    (not (not-embedded-in-segment-p *segment-array* pos))))

(defMethod style-strings ((buffer buffer-mark) &optional (start 0) (end (buffer-size buffer))
                           &aux string-instances)
   (setf *segment-list* (create-string-list buffer start end *dynamic-p*))
   (do* ((string-list *segment-list* (rest string-list))
          (start-string (first (first string-list)) (first (first string-list)))
          (end-string (second (first string-list)) (second (first string-list))))
         ((null start-string) string-instances)
      (push (make-instance 'styled-string
                :string-buffer buffer
                :string-start start-string
                :string-end end-string)
             string-instances))
   ;; create the segment array - if styling dynamically, 
   ;; we need to create the inclusive string list for the segment array.
   (setf *segment-array* (make-segment-array 
                          (if *dynamic-p*
                            (setf *segment-list* (create-string-list buffer start end))
                            *segment-list*)))
   (dolist (string string-instances)
      (style-string string))
   string-instances)

(defMethod style-semi-colon-comments ((buffer buffer-mark)
                                            &optional (start 0) (end (buffer-size buffer)))
   ;;Make a list of instances, Style them.
   (let ((comment-instances nil)
         (comment-segment-list nil))
     (do* ((start-comment (ccl::buffer-forward-search buffer #\; start)
                           (ccl::buffer-forward-search buffer #\; end-comment))
            (end-comment (when start-comment
                            (let ((ret (line-end buffer start-comment)))
                              (or ret end)))
                         (when start-comment
                            (let ((ret (line-end buffer start-comment)))
                              (or ret end)))))
           ((or (not start-comment)
                 (not end-comment)
                 (> start-comment end)))
        
        ;; Note: buffer-forward-search returns the position AFTER the the target char.
        (setf start-comment (1- start-comment))
        (unless (= end-comment (buffer-size buffer))
           (setf end-comment (1- end-comment)))
        ;; The first AND handles the case where a string spans two comments. 
        (when (or (and (= start-comment (buffer-line-start buffer start-comment))
                          ;; support for dynamic styling 
                          (or (not *dynamic-p*)
                               (and *dynamic-p* 
                                     (>= *dynamic-pos* start-comment)
                                     (<= (1- *dynamic-pos*) end-comment))))
                    ;; with dynamically-style-buffer-comments *segment-array* may not be there yet.
                    (and (not (embedded-in-segment-p start-comment))
                          (not (and (>= start-comment 2)
                                      (eq (buffer-char buffer (- start-comment 1)) #\\)
                                      (eq (buffer-char buffer (- start-comment 2)) #\#)))
                          (or (not *dynamic-p*)
                               (and *dynamic-p* 
                                     (>= *dynamic-pos* start-comment)
                                     (<= (1- *dynamic-pos*) end-comment)))))
           (push (list start-comment end-comment) comment-segment-list)
           (push (make-instance 'styled-semi-colon-comment 
                     :comment-buffer buffer
                     :comment-start start-comment
                     :comment-end end-comment)
                  comment-instances)))
     (setf *segment-list* 
           (unify-segment-lists (nreverse comment-segment-list) *segment-list*))
     (setf *segment-array* (make-segment-array *segment-list*))
     (dolist (comment comment-instances)
        (style-comment comment))
     comment-instances))


#| This is a comment #||# with #| numerous #|nested|# |# #|comments|# in it |# 
(defMethod style-sharp-comments ((buffer buffer-mark) 
                                      &optional (start 0) (end (buffer-size buffer)))
   ;; handle nested comments ...
   (flet ((find-end-comment (start-comment)
           (do* ((level-count 1)
                  (next-end-comment (ccl::buffer-string-pos buffer "|#" :start start-comment :end end)
                                    (when next-start-comment
                                       (ccl::buffer-string-pos buffer "|#" :start (+ next-end-comment 2) :end end)))
                  (next-start-comment (ccl::buffer-string-pos buffer "#|" :start (+ start-comment 2) :end end)
                                      (when next-start-comment
                                         (ccl::buffer-string-pos buffer "#|" :start (+ next-start-comment 2) :end end))))
                 ((null next-end-comment))
              
              (when (and next-start-comment (< next-start-comment next-end-comment))
                 ;; nested
                 (incf level-count))
              
              (decf level-count)
              (when (= level-count 0) (return next-end-comment)))))
     
     (let ((comment-instances nil)
           (comment-segment-list nil))
       (do* ((start-comment (ccl::buffer-string-pos buffer "#|" :start start)
                             (ccl::buffer-string-pos buffer "#|" :start end-comment))
              (end-comment (when (and start-comment (<= start-comment end))
                              (find-end-comment start-comment))
                           (when (and start-comment (<= start-comment end))
                              (find-end-comment start-comment))))
             ((or (not start-comment) 
                   (not end-comment)))
          (cond ((and (not-embedded-in-segment-p *segment-array* start-comment)
                        (not-embedded-in-segment-p *segment-array* end-comment)
                        ;; support for dynamic styling 
                        (or (not *dynamic-p*)
                             (and *dynamic-p* 
                                   (>= *dynamic-pos* start-comment)
                                   ;; end-comment does not include the "|#" 
                                   (<= (- *dynamic-pos* 3) end-comment))))
                   (push (list start-comment end-comment) comment-segment-list)
                   (push (make-instance 'styled-sharp-comment 
                             :comment-buffer buffer
                             :comment-start start-comment
                             :comment-end (+ 2 end-comment))
                          comment-instances))))
       (when comment-instances
          (setf *segment-list* (unify-segment-lists (nreverse comment-segment-list) *segment-list*))
          (setf *segment-array* (make-segment-array *segment-list*))
          (dolist (comment comment-instances)
             (style-comment comment))
          comment-instances))))

(defMethod view-style-buffer-comments ((fred fred-window) &optional start end)
  (unless (and start end)
    (multiple-value-setq (start end) (selection-range fred))
    (when (= start end)
      (setf start 0
            end (buffer-size (fred-buffer fred)))))
  (let* ((buffer (fred-buffer fred))
         (string-instances (style-strings buffer start end))
         (semi-colon-instances (style-semi-colon-comments buffer start end))
         (sharp-instances (style-sharp-comments buffer  start end)))
    (nconc semi-colon-instances sharp-instances string-instances)))

(defMethod view-style-buffer-comments ((fred fred-item) &optional start end)
  ;; let's not style the listener 
  (unless (typep (view-window fred) 'listener)
    (unless (and start end)
      (multiple-value-setq (start end) (selection-range fred))
      (when (= start end)
        (setf start 0
              end (buffer-size (fred-buffer fred)))))
    (let* ((buffer (fred-buffer fred))
           (string-instances (style-strings buffer start end))
           (semi-colon-instances (style-semi-colon-comments buffer start end))
           (sharp-instances (style-sharp-comments buffer start end)))
      (nconc semi-colon-instances sharp-instances string-instances))))

(defun dynamically-style-buffer-comments (fred start end position 
                                                   style-strings-p style-semi-colon-comments-p)
   (let* ((buffer (fred-buffer fred))
          (line-start (buffer-line-start buffer position))
          (string-instances (when style-strings-p 
                               (style-strings buffer start end)))
          (semi-colon-instances (when style-semi-colon-comments-p 
                                   (style-semi-colon-comments buffer line-start end))))
     (nconc semi-colon-instances string-instances)))

;;; Do-nothing for various tools using mixins.
(defMethod view-style-buffer-comments ((fred fred-mixin) &optional start end)
  (declare (ignore start end))
  ())

(defMethod get-combined-segment-list ((buffer buffer-mark))
  (let* ((size (buffer-size buffer))
         (string-list (create-string-list buffer 0 size))
         (semi-colon-comment-list (create-semi-colon-comment-list buffer 0 size))
         (sharp-stroke-comment-list (create-sharp-stroke-comment-list buffer 0 size)))
    (unify-segment-lists string-list 
                         (unify-segment-lists semi-colon-comment-list
                                              sharp-stroke-comment-list))))

;;; ----------------------------------------------------------------------------
;;; Styling menu items
;;; ----------------------------------------------------------------------------
;;;
(when (find-menu-item  *edit-menu* "Style File")
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* "Style File")))

(when (find-menu-item  *edit-menu* "Style Selection")
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* "Style Selection")))

(when (find-menu-item  *edit-menu* "Style File Vanilla")
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* "Style File Vanilla")))

(when (find-menu-item  *edit-menu* "Style Selection Vanilla")
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* "Style Selection Vanilla")))

(when (find-menu-item  *edit-menu* "Style Folder ...")
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* "Style Folder ...")))

(add-menu-items *edit-menu* (make-instance 'menu-item :menu-item-title "-"))

(add-menu-items *edit-menu*
                (make-instance 'window-menu-item
                  :menu-item-title "Style File"
                  :menu-item-action 'style-file
                  :update-function 'update-style-file
                  :command-key #\U))

(defMethod style-file ((fred fred-window))
  (eval-enqueue `(window-style-buffer ,fred)))

(defun update-style-file (item)
  (let ((w (front-window)))
    (cond ((and (typep w 'fred-window) (not (typep w 'listener)))
           (menu-item-enable item)
           (multiple-value-bind (start end)
                                (selection-range w)
             (cond ((= start end) ; no selection
                    (set-menu-item-title item "Style File"))
                   (t
                    (set-menu-item-title item "Style Selection")))))
          (t
           (menu-item-disable item)))))
  
(add-menu-items *edit-menu*
                (make-instance 'window-menu-item
                  :menu-item-title "Style File Vanilla"
                  :update-function 'update-style-file-vanilla
                  :menu-item-action 'style-file-vanilla
                  :command-key #\J))

(defMethod style-file-vanilla ((fred fred-window))
  (multiple-value-bind (start end)
                       (selection-range fred)
    (cond ((= start end) ; no selection
           (eval-enqueue 
            `(vanilla-style ,fred 0 (buffer-size (fred-buffer ,fred)))))
          (t
           (eval-enqueue `(vanilla-style ,fred ,start ,end))))))         

(defun update-style-file-vanilla (item)
  (let ((w (front-window)))
    (cond ((and (typep w 'fred-window) (not (typep w 'listener)))
           (menu-item-enable item)
           (multiple-value-bind (start end)
                                (selection-range w)
             (cond ((= start end) ; no selection
                    (set-menu-item-title item "Style File Vanilla"))
                   (t
                    (set-menu-item-title item "Style Selection Vanilla")))))
          (t
           (menu-item-disable item)))))

(add-menu-items *edit-menu*
                (make-instance 'menu-item
                  :menu-item-title "Style Folder ..."
                  :menu-item-action #'(lambda ()
                                       (eval-enqueue '(progn (style-folder-recursively)
                                                       (ed-beep))))))

;;; Styling menu items
(defParameter *style-prefs-dialog* "Style Prefs ..." "The name of the menu item.")

(when (find-menu-item  *edit-menu* *style-prefs-dialog*)
  (remove-menu-items *edit-menu*
                     (find-menu-item  *edit-menu* *style-prefs-dialog*)))

;;; add menu item:
(add-menu-items *edit-menu*
                (make-instance 'menu-item
                  :menu-item-title *style-prefs-dialog*
                  :menu-item-action 'open-style-prefs-dialog))

